#!/usr/bin/env bash

CLUSTER_NAME=""
REGION="us-east-1"
FORCE=false
TAG_KEY="cluster"
TAG_VALUE="" # Cluster name will be appended to this

# show_help prints the help for the script
function show_help() {
  echo "Usage: $0 [options]"
  echo ""
  echo "Options:"
  echo "  -c, --cluster <cluster-name>      EKS cluster name. Required."
  echo "  -r, --region <region>             AWS region (default: ${REGION})."
  echo "  -f, --force                       Force the script, i.e., skip the prompts and assume yes for all."
  echo "  -h, --help                        Display this help and usage message. Note: This doesn't exit if there's a failure. You can run multiple times to verify all resources are deleted."
}

# error prints error message to stderr and exit code 1
function error() {
    echo "Error: $@" >&2
    show_help

    exit 1
}

# parse_flags parses the input flags
function parse_flags() {
    while [[ "$#" -gt 0 ]]; do
        case "$1" in
            -c|--cluster)
                if [ -n "$2" ] && [[ "$2" != -* ]]; then
                    CLUSTER_NAME="$2"
                    shift
                else
                    error "-c/--cluster requires a value"
                fi
                ;;
            -r|--region)
                if [ -n "$2" ] && [[ "$2" != -* ]]; then
                    REGION="$2"
                    shift
                else
                    error "-r/--region requires a value"
                fi
                ;;
            -f|--force)
                FORCE=true
                ;;
            -h|--help)
                show_help
                exit 0
                ;;
            *)
                error "Unknown option: $1"
                ;;
        esac
        shift
    done

    if [[ -z "${CLUSTER_NAME}" ]]; then
        error "Missing cluster name"
    fi
    TAG_VALUE="${TAG_VALUE}${CLUSTER_NAME}"

    print_inputs=$(cat <<EOF
Cluster name: "${CLUSTER_NAME}"
Region: "${REGION}"
Force: ${FORCE}
terraform-exostellar-modules's tag: "${TAG_KEY}:${TAG_VALUE}"
EOF
    )
    
    max_length=0
    while IFS= read -r line; do
    if [ ${#line} -gt $max_length ]; then
        max_length=${#line}
    fi
    done <<< "$print_inputs"

    divider=$(printf '%*s' "$max_length" '' | tr ' ' '-')

    echo "$divider"
    printf "${print_inputs}\n"
    echo "$divider"
}

# prompt_for_clean_up throws a general warning that resources will be cleaned-up and prompts whether to proceed.
# If -f/--force is enabled, it directly proceeds without prompting. 
function prompt_for_clean_up() {
    if [[ ${FORCE} == "true" ]]; then
        printf "Force is enabled. Directly proceeding with the clean-up.\n\n"

        return
    fi

    while true; do
        read -p "WARNING: This command destroys all the resources with terraform-exostellar-modules's tag \"${TAG_KEY}:${TAG_VALUE}\" in AWS. Do you want to proceed? (y/n): " proceed
        case "${proceed}" in
            [Yy]*) 
                printf "Proceeding with the clean-up.\n\n"
                break
                ;;
            [Nn]*) 
                echo "Skipping the clean-up."
                exit 0
                ;;
            *)
                echo "Invalid option ${proceed}"
                ;;
        esac
    done
}

# clean_up_cloudformation_stacks destroys all the CloudFormation stacks starting with the input cluster name
function clean_up_cloudformation_stacks() {
    echo "Checking for CloudFormation stacks with terraform-exostellar-modules's tag \"${TAG_KEY}:${TAG_VALUE}\" in region \"${REGION}\"..."
    for stack in $(
        aws cloudformation list-stacks \
            --region "${REGION}" \
            --query "StackSummaries[?Tags && contains(Tags[?Key==\`${TAG_KEY}\`&&Value==\`${TAG_VALUE}\`].Value, \`${TAG_VALUE}\`)].[StackName]" \
            --output text \
            | cat
    ); do
        echo "Deleting CloudFormation Stack: ${stack}"
        aws cloudformation delete-stack \
            --region "${REGION}" \
            --stack-name "${stack}" \
            | cat
    done
}

# clean_up_eks_clusters destroys all the EKS clusters starting with the input cluster name
function clean_up_eks_clusters() {
    echo "Checking for EKS clusters with terraform-exostellar-modules's tag \"${TAG_KEY}:${TAG_VALUE}\" in region \"${REGION}\"..."
    for cluster in $(
        aws resourcegroupstaggingapi get-resources \
            --region "${REGION}" \
            --resource-type-filters "eks:cluster" \
            --tag-filters Key="${TAG_KEY}",Values="${TAG_VALUE}" \
            --query "ResourceTagMappingList[].ResourceARN" \
            --output text \
            | cat
    ); do
        echo "Checking for nodegroups in EKS Cluster \"${cluster}\" in region \"${REGION}\"..."
        for nodegroup in $(
            aws eks list-nodegroups \
                --region "${REGION}" \
                --cluster-name $(basename "${cluster}") \
                --query "nodegroups[]" \
                --output text \
                | cat
            ); do
            echo "Deleting node group ${nodegroup} in cluster ${cluster}..."
            aws eks delete-nodegroup \
                --region "${REGION}" \
                --cluster-name $(basename "${cluster}") \
                --nodegroup-name "${nodegroup}" \
                | cat
            
            echo "Waiting for deletion of nodegroup \"${nodegroup}\"..."
            aws eks wait nodegroup-deleted \
                --region "${REGION}" \
                --cluster-name $(basename "${cluster}") \
                --nodegroup-name "${nodegroup}" \
                | cat
        done

        echo "Deleting EKS Cluster: ${cluster}"
        aws eks delete-cluster \
            --region "${REGION}" \
            --name $(basename "${cluster}") \
            | cat
        
        echo "Waiting for deletion of EKS cluster \"${cluster}\"..."
        aws eks wait cluster-deleted \
            --region "${REGION}" \
            --name $(basename "${cluster}") \
            | cat
    done
}

# clean_up_ec2_instances destroys all the EC2 instances starting with the input cluster name
function clean_up_ec2_instances() {
    echo "Checking for EC2 instances with terraform-exostellar-modules's tag \"${TAG_KEY}:${TAG_VALUE}\" in region \"${REGION}\"..."
    instances=$(
        aws ec2 describe-instances \
        --region "${REGION}" \
        --filters "Name=tag:${TAG_KEY},Values=${TAG_VALUE}" \
        --query "Reservations[*].Instances[*].{InstanceId:InstanceId,Name:Tags[?Key=='Name']|[0].Value,State:State.Name}" \
        --output json \
        | cat
    )
    for instance in $(echo "$instances" | jq -c '.[]'); do
        instance_id=$(echo "${instance}" | jq -r '.[0].InstanceId')
        instance_name=$(echo "${instance}" | jq -r '.[0].Name')
        instance_state=$(echo "${instance}" | jq -r '.[0].State')

        if [[ 
                "$instance_state" == "shutting-down" || 
                "$instance_state" == "terminated"
            ]]; then
            continue
        fi

        echo "Disabling EC2 Instance's termination protection: \"${instance_name}\" (ID: \"${instance_id}\")"
        aws ec2 modify-instance-attribute \
            --instance-id "${instance_id}" \
            --region "${REGION}" \
            --no-disable-api-termination \
            | cat
        
        echo "Deleting EC2 Instance: \"${instance_name}\" (ID: \"${instance_id}\")"
        aws ec2 terminate-instances \
            --region "${REGION}" \
            --instance-ids "${instance_id}" \
            | cat

        echo "Waiting for deletion of EC2 instance \"${instance_name}\"..."
        aws ec2 wait instance-terminated \
            --region "${REGION}" \
            --instance-ids "${instance_id}"
    done
}

# clean_up_nat_gateways destroys all the NAT gateways starting with the input cluster name
function clean_up_nat_gateways() {
    echo "Checking for NAT Gateways with terraform-exostellar-modules's tag \"${TAG_KEY}:${TAG_VALUE}\" in region \"${REGION}\"..."
    nat_gateways=$(
        aws ec2 describe-nat-gateways \
            --region "${REGION}" \
            --filter "Name=tag:${TAG_KEY},Values=${TAG_VALUE}" \
            --query "NatGateways[*].{NatGatewayId:NatGatewayId,State:State}" \
            --output json
    )
    for nat_gateway in $(echo "$nat_gateways" | jq -c '.[]'); do
        nat_gateway_id=$(echo ${nat_gateway} | jq -r '.NatGatewayId')
        state=$(echo ${nat_gateway} | jq -r '.State')

        if [[ "${state}" == 'deleted' ]]; then
            continue
        fi

        echo "Deleting NAT Gateway with ID: ${nat_gateway_id}"
        aws ec2 delete-nat-gateway \
            --region "${REGION}" \
            --nat-gateway-id "${nat_gateway_id}" \
            | cat

        aws ec2 wait nat-gateway-deleted \
            --region "${REGION}" \
            --nat-gateway-id "${nat_gateway_id}"
    done
}

# clean_up_elastic_ips destroys all the Elastic IPs starting with the input cluster name
function clean_up_elastic_ips() {
    echo "Checking for Elastic IP Address with terraform-exostellar-modules's tag \"${TAG_KEY}:${TAG_VALUE}\" in region \"${REGION}\"..."
    for eip_allocation_id in $(
        aws ec2 describe-addresses \
            --region "${REGION}" \
            --filters "Name=tag:${TAG_KEY},Values=${TAG_VALUE}" \
            --query "Addresses[*].AllocationId" \
            --output text \
            | cat
        ); do
        if [ "${allocation_id}" == "None" ]; then
            continue
        fi

        echo "Releasing Elastic IP Address: ${eip_allocation_id}"
        aws ec2 release-address \
            --region "${REGION}" \
            --allocation-id "${eip_allocation_id}" \
            | cat
    done
}

# clean_up_vpcs_and_dependent_network_resources destroys all the VPCs starting with the input cluster name and related network resources
function clean_up_vpcs_and_dependent_network_resources() {
    echo "Checking for VPC with terraform-exostellar-modules's tag \"${TAG_KEY}:${TAG_VALUE}\" and related network resources in region \"${REGION}\"..."
    vpcs=$(
        aws ec2 describe-vpcs \
            --region "${REGION}" \
            --filters "Name=tag:${TAG_KEY},Values=${TAG_VALUE}" \
            --query "Vpcs[].{VpcId:VpcId}" \
            --output json \
            | cat
    )
    for vpc in $(echo $vpcs | jq -c '.[]'); do
        vpc_id=$(echo "${vpc}" | jq -r '.VpcId')

        echo "Checking for Internet Gateways associated with VPC with ID \"${vpc_id}\"..."
        for igw in $(
            aws ec2 describe-internet-gateways \
                --region "${REGION}" \
                --filters "Name=attachment.vpc-id,Values=${vpc_id}" \
                --query "InternetGateways[].InternetGatewayId" \
                --output text \
                | cat
            ); do
            echo "Deleting Internet Gateway \"${igw}\"..."
            aws ec2 detach-internet-gateway \
                --region "${REGION}" \
                --internet-gateway-id "${igw}" \
                --vpc-id "${vpc_id}" \
                | cat
            aws ec2 delete-internet-gateway \
                --region "${REGION}" \
                --internet-gateway-id "$igw" \
                | cat
        done

        echo "Checking for Subnets associated with VPC with ID \"${vpc_id}\"..."
        for subnet in $(
            aws ec2 describe-subnets \
                --region "${REGION}" \
                --filters "Name=vpc-id,Values=${vpc_id}" \
                --query "Subnets[].SubnetId" \
                --output text \
                | cat
            ); do
            echo "Deleting Subnet \"${subnet}\"..."
            aws ec2 delete-subnet \
                --region "${REGION}" \
                --subnet-id "${subnet}" \
                | cat
        done

        echo "Checking for Route Tables associated with VPC with ID \"${vpc_id}\"..."
        for rtb in $(
            aws ec2 describe-route-tables \
                --region "${REGION}" \
                --filters "Name=vpc-id,Values=${vpc_id}" \
                --query "RouteTables[?Associations == \`[]\` || Associations[?Main == \`false\`]].RouteTableId" \
                --output text \
                | cat
            ); do
            echo "Deleting Route Table \"${rtb}\"..."
            aws ec2 delete-route-table \
                --region "${REGION}" \
                --route-table-id "${rtb}" \
                | cat
        done

        echo "Checking for Security Groups associated with VPC with ID \"${vpc_id}\"..."
        declare -A security_groups
        for group_id in $(
            aws ec2 describe-security-groups \
                --region "${REGION}" \
                --filters "Name=vpc-id,Values=${vpc_id}" \
                --query "SecurityGroups[?GroupName!='default'].GroupId" \
                --output text \
                | cat
            ); do
            security_groups["$group_id"]=1
        done

        for sg in "${!security_groups[@]}"; do
            ingressIpPermissions=$(
                aws ec2 describe-security-groups \
                    --region "${REGION}" \
                    --group-ids "${sg}" \
                    --query "SecurityGroups[0].IpPermissions" \
                    --output json \
                    | cat
            )
            if [[ "${ingressIpPermissions}" == '[]' ]]; then
                continue
            fi

            echo "Revoking Ingress Rule \"${ingressIpPermissions}\" from Security Group \"${sg}\"..."
            aws ec2 revoke-security-group-ingress \
                --region "${REGION}" \
                --group-id "${sg}" \
                --ip-permissions "${ingressIpPermissions}" \
                | cat
        done
            
        for sg in "${!security_groups[@]}"; do
            egressIpPermissions=$(
                aws ec2 describe-security-groups \
                    --region "${REGION}" \
                    --group-ids "${sg}" \
                    --query "SecurityGroups[0].IpPermissionsEgress" \
                    --output json \
                    | cat
            )
            if [[ "${egressIpPermissions}" == '[]' ]]; then
                continue
            fi

            echo "Revoking Egress Rule \"${egressIpPermissions}\" from Security Group \"${sg}\"..."
            aws ec2 revoke-security-group-egress \
                --region "${REGION}" \
                --group-id "${sg}" \
                --ip-permissions "${egressIpPermissions}" \
                | cat
        done
            
        echo "Deleting security group: ${sg}"
        aws ec2 delete-security-group \
            --region "${REGION}" \
            --group-id "${sg}" \
            | cat

        echo "Checking for Network ACLs associated with VPC with ID \"${vpc_id}\"..."
        for nacl in $(
            aws ec2 describe-network-acls \
                --region "${REGION}" \
                --filters "Name=vpc-id,Values=${vpc_id}" \
                --query "NetworkAcls[?IsDefault==\`false\`].NetworkAclId" \
                --output text \
                | cat
            ); do
            echo "Deleting Network ACL \"${nacl}\"..."
            aws ec2 delete-network-acl \
                --region "${REGION}" \
                --network-acl-id "${nacl}" \
                | cat
        done

        echo "Checking for Network Interfaces associated with VPC with ID \"${vpc_id}\"..."
        for eni in $(
            aws ec2 describe-network-interfaces \
                --region "${REGION}" \
                --filters "Name=vpc-id,Values=${vpc_id}" \
                --query "NetworkInterfaces[].NetworkInterfaceId" \
                --output text \
                | cat
            ); do
            echo "Deleting Network Interface \"${eni}\"..."
            aws ec2 delete-network-interface \
                --region "${REGION}" \
                --network-interface-id "${eni}" \
                | cat
        done

        echo "Deleting VPC with ID \"${vpc_id}\"..."
        aws ec2 delete-vpc \
            --region "${REGION}" \
            --vpc-id "${vpc_id}" \
            | cat
    done
}

# clean_up_iam_roles_and_related_resources destroys all the IAM roles starting with the input cluster name and related resources
function clean_up_iam_roles_and_related_resources() {
    echo "Checking for IAM Roles with terraform-exostellar-modules's tag \"${TAG_KEY}:${TAG_VALUE}\" and related resources..."
    role_names=$(
        aws iam list-roles \
            --query 'Roles[*].RoleName' \
            --output text \
            | cat
    )
    for role_name in ${role_names}; do
        # Check if the role name starts with the cluster name or "terraform". This is a workaround to limit the number
        # of calls to AWS for fetching tags on all available roles.
        if ! echo "${role_name}" | grep -q "^${CLUSTER_NAME}\|^terraform-"; then
            continue
        fi

        if ! aws iam list-role-tags \
            --role-name "${role_name}" \
            --query "Tags[?Key=='${TAG_KEY}' && Value=='${TAG_VALUE}']" \
            --output json | jq '.[]' | grep -q .; then
            continue
        fi

        echo "Checking for IAM resources related to role: ${role_name}"

        policy_arns=$(
            aws iam list-attached-role-policies \
                --region "${REGION}" \
                --role-name "${role_name}" \
                --query "AttachedPolicies[].PolicyArn" \
                --output text
        )
        for policy_arn in ${policy_arns}; do
            echo "Deleting policy \"${policy_arn}\" attached to role \"${role_name}\""
            aws iam detach-role-policy \
                --region "${REGION}" \
                --role-name "${role_name}" \
                --policy-arn "${policy_arn}" \
                | cat
        done

        inline_policies=$(
            aws iam list-role-policies \
                --region "${REGION}" \
                --role-name "${role_name}" \
                --query "PolicyNames" \
                --output text
        )
        for policy_name in $inline_policies; do
            echo "Deleting inline policy \"${policy_name}\" attached to role \"${role_name}\""
            aws iam delete-role-policy \
                --region "${REGION}" \
                --role-name "${role_name}" \
                --policy-name "${policy_name}" \
                | cat
        done

        instance_profiles=$(
            aws iam list-instance-profiles-for-role \
                --region "${REGION}" \
                --role-name "${role_name}" \
                --query "InstanceProfiles[].InstanceProfileName" \
                --output text
        )
        for instance_profile in ${instance_profiles}; do
            echo "Deleting instance profile \"${instance_profile}\" attached to role \"${role_name}\""
            aws iam remove-role-from-instance-profile \
                --region "${REGION}" \
                --instance-profile-name "${instance_profile}" \
                --role-name "${role_name}" \
                | cat
        done

        echo "Deleting role: ${role_name}"
        aws iam delete-role \
            --region "${REGION}" \
            --role-name "${role_name}" \
            | cat
    done
}

# clean_up_individual_iam_policies destroys all the individual IAM policies starting with the input cluster name
function clean_up_individual_iam_policies() {
    echo "Checking for individual IAM Policies with terraform-exostellar-modules's tag \"${TAG_KEY}:${TAG_VALUE}\" and their versions..."
    policies=$(
        aws iam list-policies \
            --query 'Policies[*].{PolicyName:PolicyName,Arn:Arn}' \
            --output json \
            | cat
    )
    for policy in $(echo "${policies}" | jq -c '.[]'); do
        policy_name=$(echo "${policy}" | jq -r '.PolicyName')
        policy_arn=$(echo "${policy}" | jq -r '.Arn')
        # Check if the policy's name starts with the cluster name or "AmazonEKS_EBS_CSI_Policy-". This is a workaround
        # to limit the number of calls to AWS for fetching tags on all available policies.
        if ! echo "${policy_name}" | grep -q "^${CLUSTER_NAME}\|^AmazonEKS_EBS_CSI_Policy-"; then
            continue
        fi

        if ! aws iam list-policy-tags \
            --policy-arn "${policy_arn}" \
            --query "Tags[?Key=='${TAG_KEY}' && Value=='${TAG_VALUE}']" \
            --output json | jq '.[]' | grep -q .; then
            continue
        fi

        echo "Checking for versions of policy: ${policy_name}"

        version_ids=$(
            aws iam list-policy-versions \
                --region "${REGION}" \
                --policy-arn "${policy_arn}" \
                --query "Versions[?IsDefaultVersion==\`false\`].VersionId" \
                --output text
        )
        for version_id in ${version_ids}; do
            aws iam delete-policy-version \
                --region "${REGION}" \
                --policy-arn "${policy_arn}" \
                --version-id "${version_id}" \
                | cat
        done

        echo "Deleting policy: ${policy_arn}"
        aws iam delete-policy \
            --region "${REGION}" \
            --policy-arn "${policy_arn}" \
            | cat
    done
}

# clean_up_individual_iam_instance_profiles destroys all the individual IAM Instance Policies starting with the input cluster name
function clean_up_individual_iam_instance_profiles() {
    echo "Checking for individual IAM Instance Profiles with terraform-exostellar-modules's tag \"${TAG_KEY}:${TAG_VALUE}\" and related resources..."
    instance_profile_names=$(
        aws iam list-instance-profiles \
            --query 'InstanceProfiles[].InstanceProfileName' \
            --output text \
            | cat
    )
    for instance_profile_name in ${instance_profile_names}; do
        # Check if the instance profile's name starts with the cluster name. This is a workaround to limit the number
        # of calls to AWS for fetching tags on all available instance profiles.
        if ! echo "${instance_profile_name}" | grep -q "^${CLUSTER_NAME}"; then
            continue
        fi

        if ! aws iam list-instance-profile-tags \
            --instance-profile-name "${instance_profile_name}" \
            --query "Tags[?Key=='${TAG_KEY}' && Value=='${TAG_VALUE}']" \
            --output json | jq '.[]' | grep -q .; then
            continue
        fi

        echo "Checking for IAM roles attached to instance profile: ${instance_profile_name}"

        role_names=$(
            aws iam get-instance-profile \
                --region "${REGION}" \
                --instance-profile-name "${instance_profile_name}" \
                --query "InstanceProfile.Roles[].RoleName" \
                --output text
        )
        for role_name in ${role_names}; do
            aws iam remove-role-from-instance-profile \
                --region "${REGION}" \
                --instance-profile-name "${instance_profile_name}" \
                --role-name "${role_name}" \
                | cat
        done

        echo "Deleting instance profile: ${instance_profile_name}"
        aws iam delete-instance-profile \
            --region "${REGION}" \
            --instance-profile-name "${instance_profile_name}" \
            | cat
    done
}

# clean_up_cloudwatch_log_groups destroys all the CloudWatch Log Groups starting with the input cluster name
function clean_up_cloudwatch_log_groups() {
    echo "Checking for CloudWatch Log Groups with terraform-exostellar-modules's tag \"${TAG_KEY}:${TAG_VALUE}\" in region \"${REGION}\"..."
    log_group_names=$(
        aws resourcegroupstaggingapi get-resources \
            --region "${REGION}" \
            --resource-type-filters "logs:log-group" \
            --tag-filters Key="${TAG_KEY}",Values="${TAG_VALUE}" \
            --query "ResourceTagMappingList[].ResourceARN" \
            --output text \
            | cat
    )
    for log_group_name in ${log_group_names}; do
        echo "Deleting log group: ${log_group_name}"
        aws logs delete-log-group \
            --region "${REGION}" \
            --log-group-name "$(echo ${log_group_name} | cut -d':' -f7)" \
            | cat
    done
}

# clean_up_kms_customer_managed_key_aliases destroys all the KMS Customer Managed Key's Aliases starting with the input cluster name
function clean_up_kms_customer_managed_key_aliases() {
    echo "Checking for KMS Customer Managed Key's Aliases with terraform-exostellar-modules's tag \"${TAG_KEY}:${TAG_VALUE}\"..."
    aliases=$(
        aws kms list-aliases \
            --query 'Aliases[?length(AliasName) > `0` && !starts_with(AliasName, `alias/aws/`)].{AliasName:AliasName,AliasID:TargetKeyId}' \
            --output json \
            | cat
    )
    for alias in $(echo "${aliases}" | jq -c '.[]'); do
        alias_name=$(echo "${alias}" | jq -r '.AliasName')
        alias_id=$(echo "${alias}" | jq -r '.AliasID')
        # Check if the KMS alias' name ends with the cluster name. This is a workaround to limit the number of calls to
        # AWS for fetching tags on all available KMS aliases.
        if ! echo "${alias_name}" | grep -q "${CLUSTER_NAME}$"; then
            continue
        fi

        if ! aws kms list-resource-tags \
            --key-id "${alias_id}" \
            --query "Tags[?TagKey=='${TAG_KEY}' && TagValue=='${TAG_VALUE}']" \
            --output json | jq '.[]' | grep -q .; then
            continue
        fi

        echo "Deleting alias: ${alias_name}"
        aws kms delete-alias \
            --region "${REGION}" \
            --alias-name "${alias_name}" \
            | cat
    done
}

# Setup
parse_flags $@
prompt_for_clean_up

# Clean-up
clean_up_cloudformation_stacks
clean_up_eks_clusters
clean_up_ec2_instances
clean_up_nat_gateways
clean_up_elastic_ips
clean_up_vpcs_and_dependent_network_resources
clean_up_iam_roles_and_related_resources
clean_up_individual_iam_policies
clean_up_individual_iam_instance_profiles
clean_up_cloudwatch_log_groups
clean_up_kms_customer_managed_key_aliases
