module "xio_infra" {
  source = "../infra"

  aws_region  = var.aws_region
  eks_cluster = var.eks_cluster
  eks_version = var.eks_version
  vpc_cidr    = var.vpc_cidr
}

resource "null_resource" "update_kubeconfig" {
  depends_on = [module.xio_infra]

  provisioner "local-exec" {
    command = "aws eks update-kubeconfig --region ${var.aws_region} --name ${var.eks_cluster}"
  }
}

module "exostellar_iam" {
  source     = "../iam"
  depends_on = [module.xio_infra, null_resource.update_kubeconfig]

  aws_region  = var.aws_region
  eks_cluster = var.eks_cluster
}

module "exostellar_management_server" {
  source     = "../ems"
  depends_on = [module.xio_infra, null_resource.update_kubeconfig, module.exostellar_iam]

  # Use common values like cluster name and AWS region from IAM module's outputs so EMS waits for IAM module's
  # deployment.
  eks_cluster      = var.eks_cluster
  aws_region       = var.aws_region
  vpc_id           = module.xio_infra.vpc_id
  subnet_id        = module.xio_infra.public_subnets[0]
  is_subnet_public = true # By the design of "infra" module, the subnet ID provided will be public.
  shared_security_group_ids = concat(
    [module.xio_infra.eks_cluster_security_group_id],
    module.xio_infra.eks_cluster_additional_security_group_ids,
  )
  xspot_controller_subnet_id            = module.xio_infra.private_subnets[0]
  xspot_controller_instance_profile_arn = module.exostellar_iam.xspot_controller_instance_profile_arn
  xspot_worker_subnet_id                = module.xio_infra.private_subnets[0]
  xspot_worker_instance_profile_arn     = module.exostellar_iam.xspot_worker_instance_profile_arn
  environment_image_name                = format("k8s-%s", replace(module.xio_infra.eks_version, ".", ""))

  ssh_key_name = var.ssh_key_name
  region_ami_map = {
    (var.aws_region) : var.ems_ami_id
  }
  termination_protection      = var.ems_termination_protection
  xspot_enable_hyperthreading = var.xspot_enable_hyperthreading
  xspot_enable_balloon        = var.xspot_enable_balloon
  profile_az                  = var.profile_availability_zone
  domain_name                 = var.domain_name
  instance_type               = var.ems_instance_type
  volume_size                 = var.ems_volume_size_gb
  nfs_dns_name                = var.nfs_dns_name
  nfs_security_group_id       = var.nfs_security_group_id
}

module "xkarpenter" {
  source     = "../karpenter"
  depends_on = [module.xio_infra, null_resource.update_kubeconfig, module.exostellar_iam, module.exostellar_management_server]

  # Use common values like cluster name from EMS module's outputs so xkarpenter waits for EMS module's deployment.
  eks_cluster                             = var.eks_cluster
  exostellar_management_server_private_ip = module.exostellar_management_server.exostellar_management_server_private_ip
  exostellar_management_server_port       = var.exostellar_management_server_port
  helm_chart_repository                   = var.xkarpenter_helm_chart_repository
  namespace                               = var.xkarpenter_namespace
  pod_resources                           = var.pod_resource_limits
  xkarpenter_version                      = var.xkarpenter_version
  xspot_controller_instance_profile_arn   = module.exostellar_iam.xspot_controller_instance_profile_arn
  xspot_worker_instance_profile_arn       = module.exostellar_iam.xspot_worker_instance_profile_arn
}
