locals {
  vpc_azs = tolist(
    compact(
      # Availability zone us-east-1e throws UnsupportedAvailabilityZoneException. Excluded if exists in list.
      [for az in data.aws_availability_zones.available.names : az if az != "us-east-1e"]
    )
  )

  module_tags = {
    cluster = var.eks_cluster
    region  = var.aws_region
  }
}

data "aws_availability_zones" "available" {
  state = "available"
}

module "vpc" {
  source  = "terraform-aws-modules/vpc/aws"
  version = "~> 5.13"

  name            = var.eks_cluster
  cidr            = var.vpc_cidr
  azs             = local.vpc_azs
  private_subnets = [for k, v in local.vpc_azs : cidrsubnet(var.vpc_cidr, 6, k)]
  public_subnets  = [for k, v in local.vpc_azs : cidrsubnet(var.vpc_cidr, 6, k + 6)]

  enable_dns_hostnames    = true
  enable_dns_support      = true
  enable_nat_gateway      = true
  single_nat_gateway      = true
  map_public_ip_on_launch = true

  tags                          = local.module_tags
  customer_gateway_tags         = local.module_tags
  database_acl_tags             = local.module_tags
  database_route_table_tags     = local.module_tags
  database_subnet_group_tags    = local.module_tags
  database_subnet_tags          = local.module_tags
  default_network_acl_tags      = local.module_tags
  default_route_table_tags      = local.module_tags
  default_security_group_tags   = local.module_tags
  default_vpc_tags              = local.module_tags
  dhcp_options_tags             = local.module_tags
  elasticache_acl_tags          = local.module_tags
  elasticache_route_table_tags  = local.module_tags
  elasticache_subnet_group_tags = local.module_tags
  elasticache_subnet_tags       = local.module_tags
  igw_tags                      = local.module_tags
  intra_acl_tags                = local.module_tags
  intra_route_table_tags        = local.module_tags
  intra_subnet_tags             = local.module_tags
  nat_eip_tags                  = local.module_tags
  nat_gateway_tags              = local.module_tags
  outpost_acl_tags              = local.module_tags
  outpost_subnet_tags           = local.module_tags
  private_acl_tags              = local.module_tags
  private_route_table_tags      = local.module_tags
  private_subnet_tags           = local.module_tags
  public_acl_tags               = local.module_tags
  public_route_table_tags       = local.module_tags
  public_subnet_tags            = local.module_tags
  redshift_acl_tags             = local.module_tags
  redshift_route_table_tags     = local.module_tags
  redshift_subnet_group_tags    = local.module_tags
  redshift_subnet_tags          = local.module_tags
  vpc_flow_log_tags             = local.module_tags
  vpc_tags                      = local.module_tags
  vpn_gateway_tags              = local.module_tags
}

module "eks" {
  source  = "terraform-aws-modules/eks/aws"
  version = "~> 20.0"

  cluster_name                   = var.eks_cluster
  cluster_version                = var.eks_version
  cluster_endpoint_public_access = true

  cluster_addons = {
    coredns                = {}
    eks-pod-identity-agent = {}
    kube-proxy             = {}
    vpc-cni                = {}
    aws-ebs-csi-driver = {
      service_account_role_arn = module.ebs_csi_irsa_role.iam_role_arn
    }
  }

  vpc_id                                = module.vpc.vpc_id
  subnet_ids                            = concat(module.vpc.public_subnets, module.vpc.private_subnets)
  cluster_additional_security_group_ids = [module.vpc.default_security_group_id]

  tags                           = local.module_tags
  node_iam_role_tags             = local.module_tags
  cloudwatch_log_group_tags      = local.module_tags
  cluster_encryption_policy_tags = local.module_tags
  cluster_security_group_tags    = local.module_tags
  cluster_tags                   = local.module_tags
  iam_role_tags                  = local.module_tags
  eks_managed_node_groups = {
    "${var.eks_cluster}" = {
      # Starting on 1.30, AL2023 is the default AMI type for EKS managed node groups
      ami_type                                     = "AL2023_x86_64_STANDARD"
      instance_types                               = ["m5.xlarge"]
      node_security_group_enable_recommended_rules = true

      min_size     = 1
      max_size     = 3
      desired_size = 3

      tags = local.module_tags

      iam_role_additional_policies = {
        AmazonSSMManagedInstanceCore = "arn:aws:iam::aws:policy/AmazonSSMManagedInstanceCore"
        AmazonEBSCSIDriverPolicy     = "arn:aws:iam::aws:policy/service-role/AmazonEBSCSIDriverPolicy"
      }
    }
  }

  enable_cluster_creator_admin_permissions = true

  cluster_timeouts = {
    create = "30m"
    delete = "30m"
  }
}

module "ebs_csi_irsa_role" {
  source = "terraform-aws-modules/iam/aws//modules/iam-role-for-service-accounts-eks"

  role_name             = "${var.eks_cluster}-ebs-csi-role"
  attach_ebs_csi_policy = true

  oidc_providers = {
    ex = {
      provider_arn               = module.eks.oidc_provider_arn
      namespace_service_accounts = ["kube-system:ebs-csi-controller-sa"]
    }
  }

  tags = local.module_tags
}
