locals {
  module_tags = {
    cluster = var.eks_cluster
    region  = var.aws_region
  }
}

data "aws_vpc" "cluster_vpc" {
  id = var.vpc_id
}

data "aws_subnet" "cluster_subnet" {
  id = var.subnet_id
}

data "aws_iam_policy_document" "instance_assume_role_policy" {
  statement {
    actions = ["sts:AssumeRole"]
    principals {
      type        = "Service"
      identifiers = ["ec2.amazonaws.com"]
    }
  }
}


resource "aws_security_group" "exostellar_management_server" {
  description = "Security Group for the Exostellar Management Server"
  vpc_id      = var.vpc_id
  tags = merge(
    local.module_tags,
    {
      Name = "${var.eks_cluster}-exostellar-management-server",
    }
  )

  ingress {
    from_port   = 22
    to_port     = 22
    protocol    = "tcp"
    cidr_blocks = ["0.0.0.0/0"]
  }

  ingress {
    from_port   = 443
    to_port     = 443
    protocol    = "tcp"
    cidr_blocks = ["0.0.0.0/0"]
  }

  ingress {
    from_port   = 5000
    to_port     = 5000
    protocol    = "tcp"
    cidr_blocks = [data.aws_vpc.cluster_vpc.cidr_block]
  }

  ingress {
    from_port = 0
    to_port   = 0
    protocol  = "-1"
    self      = true
  }

  egress {
    from_port   = 0
    to_port     = 0
    protocol    = "-1"
    cidr_blocks = ["0.0.0.0/0"]
  }
}

resource "aws_iam_role" "exostellar_management_server" {
  assume_role_policy = data.aws_iam_policy_document.instance_assume_role_policy.json
  tags               = local.module_tags

}

resource "aws_iam_role_policy" "exostellar_management_server" {
  name   = "${var.eks_cluster}-exostellar-management-server"
  role   = aws_iam_role.exostellar_management_server.id
  policy = file("${path.module}/policy/exostellar-management-server.json")
}

resource "aws_iam_instance_profile" "exostellar_management_server" {
  name = "${var.eks_cluster}-exostellar-management-server"
  role = aws_iam_role.exostellar_management_server.name
  tags = local.module_tags
}

resource "aws_instance" "exostellar_management_server" {
  ami                         = var.region_ami_map[var.aws_region]
  instance_type               = var.instance_type
  key_name                    = var.ssh_key_name
  subnet_id                   = var.subnet_id
  iam_instance_profile        = aws_iam_instance_profile.exostellar_management_server.name
  associate_public_ip_address = var.is_subnet_public
  vpc_security_group_ids = concat(
    [aws_security_group.exostellar_management_server.id],
    var.shared_security_group_ids != [] ? var.shared_security_group_ids : [],
    var.nfs_security_group_id != "" ? [var.nfs_security_group_id] : [],
  )

  root_block_device {
    volume_size           = var.volume_size
    delete_on_termination = true
    volume_type           = "gp2"
  }

  tags = merge(
    local.module_tags,
    {
      Name = "${var.eks_cluster}-exostellar-management-server",
    },
  )
  volume_tags = merge(
    local.module_tags,
    {
      Name = "${var.eks_cluster}-exostellar-management-server",
    },
  )

  disable_api_termination = var.termination_protection
  user_data = base64encode(
    <<-EOF
    #!/bin/bash
    cat <<EOT >> /xcompute/ems-options
      DOMAIN_NAME=${var.domain_name}
      NFS_REMOTE_HOST=${var.nfs_dns_name}
    EOT

    mkdir -p /xcompute/import/
    cat > /xcompute/import/profile_overrides.json <<JSON
    ${templatefile("${path.module}/import/profile_overrides.json", {
    profile_az                            = var.profile_az,
    profile_name                          = "az1",
    xspot_controller_instance_profile_arn = var.xspot_controller_instance_profile_arn,
    xspot_controller_security_group_ids = concat(
      [aws_security_group.exostellar_management_server.id],
      var.shared_security_group_ids,
    ),
    xspot_controller_subnet_id        = var.xspot_controller_subnet_id,
    aws_region                        = var.aws_region,
    xspot_worker_instance_profile_arn = var.xspot_worker_instance_profile_arn,
    xspot_worker_subnet_id            = var.xspot_worker_subnet_id,
    xspot_worker_security_group_ids = concat(
      [aws_security_group.exostellar_management_server.id],
      var.shared_security_group_ids,
    ),
    xspot_enable_hyperthreading = var.xspot_enable_hyperthreading,
    xspot_enable_balloon        = var.xspot_enable_balloon,
    volume_size                 = var.volume_size
    })}
    JSON

    cat > /xcompute/import/env_overrides.json <<JSON
    ${templatefile("${path.module}/import/env_overrides.json", {
    environment_image_name = var.environment_image_name
})}
    JSON
    EOF
)
}
