# Exostellar Terraform Modules

Modules for deploying Exostellar resources using Terraform.

## Usage

### 1. Standalone Flow

For deploying EKS cluster and all Exostellar resources on it, i.e., standalone flow:

```hcl
module "standalone_flow" {
  source = "../../modules/standalone-full"


  # Inputs for deploying EKS cluster and related resources.
  eks_cluster = "my-exostellar-cluster" # Name of the EKS cluster to be created.
  aws_region  = "us-east-2"             # AWS region to deploy all the resources in.
  vpc_cidr    = "10.0.0.0/16"           # CIDR block for VPC to be created, for EKS.
  eks_version = "1.31"                  # EKS (Kubernetes) version.

  # Exostellar Management Server (EMS) configurations.
  ssh_key_name               = "my-exostellar-key"     # SSH key-pair name in AWS.
  ems_ami_id                 = "ami-0adf23061c1510b3e" # EMS AMI for the selected aws_region.
  ems_instance_type          = "m5d.xlarge"            # EMS' EC2 instance type.
  ems_volume_size_gb         = 100                     # Volume size in GB.
  ems_termination_protection = true                    # EMS' EC2 instance's termination protection.
  profile_availability_zone  = "us-east-2a"            # Profile's availability zone.

  # Xspot configurations.
  xspot_enable_hyperthreading = true # Enable hyperthreading in xspot?
  xspot_enable_balloon        = true # Enable self-ballooning in xspot?

  # Exostellar Karpenter (xKarpenter) configurations.
  xkarpenter_helm_chart_repository = "oci://public.ecr.aws/u8h5n6o4/exostellar-karpenter/karpenter" # xKarpenter's Helm chart's repository.
  xkarpenter_version               = "v2.0.2"                                                       # xKarpenter's version.
  xkarpenter_namespace             = "exokarpenter"                                                 # xKarpenter's namespace.
}
```

Refer to example [standalone-flow](./examples/standalone-flow/README.md) for more details.

### 2. Existing Cluster Flow

For deploying all resources of Exostellar on an existing EKS cluster:

```hcl
module "existing_cluster_flow" {
  source = "../../modules/existing-cluster-full"


  # Inputs for deploying EKS cluster and related resources.
  eks_cluster = "my-exostellar-cluster" # Name of the existing EKS cluster.
  aws_region  = "us-east-2"             # AWS region of the EKS cluster. Default is us-east-1.
  eks_version = "1.31"                  # Kubernetes version of the existing EKS cluster. Default is 1.31.
  vpc_id      = "vpc-0b35269f469dbdc2e" # VPC ID of the EKS cluster.

  # Exostellar Management Server (EMS) configurations.
  ssh_key_name               = "my-exostellar-key"                              # SSH key-pair name in AWS.
  public_subnet_id           = "subnet-0231c16462f423e72"                       # Public subnet ID to be used with Exostellar Management Server (EMS). Note: public_subnet_id, private_subnet_id and profile_availability_zone must be of same AZ.
  profile_availability_zone  = "us-east-2a"                                     # Exostellar profile's availability zone. Note: public_subnet_id, private_subnet_id and profile_availability_zone must be of same AZ. Default is us-east-1a.
  ems_ami_id                 = "ami-0adf23061c1510b3e"                          # EMS AMI for the selected aws_region.
  ems_instance_type          = "m5d.xlarge"                                     # EMS' EC2 instance type. Default is m5d.xlarge.
  ems_volume_size_gb         = 100                                              # Volume size in GB. Default is 100 GB.
  ems_termination_protection = true                                             # EMS' EC2 instance's termination protection. Default is true.
  ems_security_group_ids     = ["sg-0d178cb07f6534981", "sg-09dfd47c24d4b36df"] # Security group IDs of existing cluster and its node group for sharing with EMS and xspot. Include any other security group IDs if required.

  # Xspot configurations.
  xspot_enable_hyperthreading = true                       # Enable hyperthreading in xspot? Default is true.
  xspot_enable_balloon        = true                       # Enable self-ballooning in xspot? Default is true.
  private_subnet_id           = "subnet-0e1a97deface1d066" # Private subnet ID to be used with xspot controllers and workers. Note: public_subnet_id, private_subnet_id and profile_availability_zone must be of same AZ.

  # Exostellar Karpenter (xKarpenter) configurations.
  xkarpenter_version = "v2.0.2" # xKarpenter's version. Default is v2.0.2.
}
```

Refer to example [existing-cluster-flow](./examples/existing-cluster-flow/README.md) for more details.

### Clean-up

:warning: Caution: The script deletes all the resources related to the session, standalone or existing-cluster flow.

All the resources deployed using the current modules will have a tag unique to the session: `cluster:<cluster-name>`. This is the same for both standalone and existing cluster flows. The [clean-up script](scripts/clean-up-exostellar-module-resources.sh) uses this tag to identify the resources and delete them.

```
./scripts/clean-up-exostellar-module-resources.sh --cluster "my-exostellar-cluster" --region "us-east-2"
```

Run it multiple times until it outputs no deletion logs. The successful output looks as follows:

```
-------------------------------------------------------------------
Cluster name: "my-exostellar-cluster"
Region: "us-east-2"
Force: false
terraform-exostellar-modules's tag: "cluster:my-exostellar-cluster"
-------------------------------------------------------------------
WARNING: This command destroys all the resources with terraform-exostellar-modules's tag "cluster:my-exostellar-cluster" in AWS. Do you want to proceed? (y/n): y
Proceeding with the clean-up.

Checking for CloudFormation stacks with terraform-exostellar-modules's tag "cluster:my-exostellar-cluster" in region "us-east-2"...
Checking for EKS clusters with terraform-exostellar-modules's tag "cluster:my-exostellar-cluster" in region "us-east-2"...
Checking for EC2 instances with terraform-exostellar-modules's tag "cluster:my-exostellar-cluster" in region "us-east-2"...
Checking for NAT Gateways with terraform-exostellar-modules's tag "cluster:my-exostellar-cluster" in region "us-east-2"...
Checking for Elastic IP Address with terraform-exostellar-modules's tag "cluster:my-exostellar-cluster" in region "us-east-2"...
Checking for VPC with terraform-exostellar-modules's tag "cluster:my-exostellar-cluster" and related network resources in region "us-east-2"...
Checking for IAM Roles with terraform-exostellar-modules's tag "cluster:my-exostellar-cluster" and related resources...
Checking for individual IAM Policies with terraform-exostellar-modules's tag "cluster:my-exostellar-cluster" and their versions...
Checking for individual IAM Instance Profiles with terraform-exostellar-modules's tag "cluster:my-exostellar-cluster" and related resources...
Checking for CloudWatch Log Groups with terraform-exostellar-modules's tag "cluster:my-exostellar-cluster" in region "us-east-2"...
Checking for KMS Customer Managed Key's Aliases with terraform-exostellar-modules's tag "cluster:my-exostellar-cluster"...
```

## Examples

- [standalone-flow](./examples/standalone-flow/README.md) - This deploys an EKS cluster with related resources, and
deploys all Exostellar resources on it.
- [existing-cluster-flow](./examples/existing-cluster-flow/README.md) - This deploys all Exostellar resources on an
existing cluster.

## Version Matrix

Following is the mapping between different components of Exostellar. It is recomended to upgrade to the latest `terraform-exostellar-modules` version and corresponding components.

| terraform-exostellar-modules | Exostellar Management Server (EMS) | Xspot Controller | Xspot Worker | Exostellar's Karpenter (xkarpenter) |
|:----------------------------:|:----------------------------------:|:----------------:|:------------:|:-----------------------------------:|
| v0.0.1 | v2.2.2+ <br>(Exostellar Release 2.5.2) | v3.2.1+ <br>(Exostellar Release 2.5.2) | v3.2.1+ <br>(Exostellar Release 2.5.2) | v2.0.2+ |

Refer to [release information page](https://exostellar.atlassian.net/wiki/spaces/ENG/pages/314638358/EAR+Non-Marketplace+Release+Information) for AMI details of Exostellar Management Server (EMS), Xspot Controller and worker.

## Authors

This module is maintained by [Exostellar][Exostellar].

## License

All rights reserved. Copyright 2023 [Exostellar Inc][Exostellar homepage].

---

[Exostellar]: https://github.com/exostellar/
[Exostellar homepage]: https://exostellar.io/
