locals {
  module_tags = {
    cluster = var.eks_cluster
    region  = var.aws_region
  }
}

# Create security group for xspot (controllers and workers).
#
# Configures two inbound rules:
# 1. Permits inbound TCP traffic on ports 32768-60999 (the default Linux ephemeral port range) from any IPv4 address,
#    including external sources outside the VPC.
# 2. Permits all inbound traffic, across all protocols and ports, from any source within the VPC.
#
# Note: The security group rules in this are only required by the xspot workers. But due to the design of xKarpenter,
# the security group is assigned to both controllers and workers. Hence, considering this as a common xspot security
# group.
resource "aws_security_group" "xspot_security_group" {
  name        = "${var.aws_resource_prefix}${var.eks_cluster}-xspot-security-group"
  description = "Security Group for the Exostellar xspot (controllers and workers)"
  vpc_id      = var.vpc_id
  tags = merge(
    local.module_tags,
    {
      Name = "${var.aws_resource_prefix}${var.eks_cluster}-xspot-security-group",
    }
  )

  # Permits inbound TCP traffic on ports 32768-60999 (the default Linux ephemeral port range) from any IPv4 address,
  # including external sources outside the VPC.
  ingress {
    from_port   = 32768         # Start of default ephemeral port range on Linux kernel.
    to_port     = 60999         # End of default ephemeral port range on Linux kernel.
    protocol    = "tcp"         # Only TCP protocol.
    cidr_blocks = ["0.0.0.0/0"] # From anywhere.
  }

  # Permits all inbound traffic, across all protocols and ports, from any source within the VPC.
  ingress {
    from_port   = 0                    # All ports.
    to_port     = 0                    # All ports.
    protocol    = "-1"                 # All protocols.
    cidr_blocks = [var.vpc_cidr_block] # Inside the current VPC.
  }
}
