##########################
### EKS Cluster Inputs ###
##########################

variable "aws_region" {
  description = "AWS region for deploying resources."
  type        = string
  default     = "us-east-1"
}

variable "eks_cluster" {
  description = "Name of the EKS cluster to be created. It must be between 1 and 26 characters in length."
  type        = string

  # EKS cluster name length validation. It must be between 1 and 26 characters.
  validation {
    condition = (
      length(var.eks_cluster) > 0 &&
      length(var.eks_cluster) <= 26
    )
    error_message = <<EOT
EKS cluster name's length must be between 1 and 26 characters in the Exostellar modules. As the EKS cluster name will be
prefixed to all Exostellar resources, due to the length limitations on a few of the resources, the EKS cluster name must
be at most 26 characters long.
EOT
  }
}

variable "eks_version" {
  description = "Version of the EKS cluster to be created."
  type        = string
  default     = "1.32"
}

variable "eks_node_group_desired_size" {
  description = "Desired size of the default EKS node group."
  type        = number
  default     = 3
}

variable "vpc_cidr" {
  description = "CIDR for VPC to be created."
  type        = string
  default     = "10.0.0.0/16"
}

#################################################
### Exostellar Management Server (EMS) Inputs ###
#################################################

variable "domain_name" {
  description = "Domain name to run Exostellar Management Server (EMS) in. Optional."
  type        = string
  default     = ""
}

variable "ssh_key_name" {
  description = "SSH key-pair's name in AWS for Exostellar Management Server (EMS)'s access. If ssm_enabled is set to true, this will be skipped."
  type        = string
  default     = ""
}

variable "ems_ami_id" {
  description = "EMS AMI for the selected aws_region."
  type        = string
}

variable "ems_instance_type" {
  description = "EC2 instance type for Exostellar Management Server (EMS)."
  type        = string
  default     = "m5d.xlarge"
}

variable "ems_volume_size_gb" {
  description = "Exostellar Management Server (EMS) instance's root volume size (in GB)"
  type        = number
  default     = 100
}

variable "profile_availability_zone" {
  description = "Exostellar profile's availability zone."
  type        = string
  default     = ""
}

variable "ems_termination_protection" {
  description = "Enable termination protection for Exostellar Management Server (EMS)'s EC2 instance?"
  type        = bool
  default     = true
}

variable "ems_volume_delete_on_termination" {
  description = "Delete the Exostellar Management Server (EMS) instance's root volume on termination of instance?"
  type        = bool
  default     = true
}

variable "ems_volume_encryption" {
  description = "Encrypt the Exostellar Management Server (EMS) instance's root volume?"
  type        = bool
  default     = false
}

variable "ems_volume_type" {
  description = "Exostellar Management Server (EMS) instance's root volume type. Valid options: standard, gp2, gp3, io1, io2, sc1, and st1."
  type        = string
  default     = "gp2"
}

variable "nfs_dns_name" {
  description = "DNS name of the NFS file system for Exostellar Management Server (EMS)."
  type        = string
  default     = ""
}

variable "nfs_security_group_id" {
  description = "Security group ID that enable traffic between the NFS server and Exostellar Management Server (EMS)."
  type        = string
  default     = ""
}

variable "license_filepath" {
  description = "The Exostellar's license file name/path."
  type        = string
  # Make this optional as the license module will be skipped if this is empty.
  default = ""
}

####################
### Xspot Inputs ###
####################

variable "xspot_enable_hyperthreading" {
  description = "Enable hyperthreading in Xspot?"
  type        = bool
  default     = true
}

variable "xspot_enable_balloon" {
  description = "Enable ballooning in Xspot?"
  type        = bool
  default     = true
}

variable "xspot_version" {
  description = "Xspot (controller and worker) version to use."
  type        = string
  default     = "xspot-3.4.0"
}

variable "allow_xspot_worker_inbound_traffic" {
  description = "Enable inbound traffic to x-spot worker?"
  type        = bool
  default     = true
}

# Enable Infrastructure Optimizer (IO) for xspot? Defaults to true.
variable "enable_infrastructure_optimizer" {
  description = "Enable Infrastructure Optimizer (IO) for xspot?"
  type        = bool
  default     = true
}

# Enable Workload Optimizer (WO) for xspot? Defaults to true.
variable "enable_workload_optimizer" {
  description = "Enable Workload Optimizer (WO) for xspot?"
  type        = bool
  default     = true
}

################################################
### Exostellar Karpenter (xKarpenter) Inputs ###
################################################

variable "xkarpenter_helm_chart_repository" {
  description = "Exostellar's Karpenter (xkarpenter)'s Helm chart's repository URL."
  type        = string
  default     = "oci://public.ecr.aws/u8h5n6o4/exostellar-karpenter/karpenter"
}

variable "xkarpenter_resources_helm_chart_repository" {
  description = "Exostellar's Karpenter (xkarpenter)'s resources (ExoNodeClass and ExoNodePool) Helm chart's repository URL."
  type        = string
  default     = "oci://public.ecr.aws/u8h5n6o4/exostellar-karpenter/karpenter-resources"
}

variable "xkarpenter_namespace" {
  description = "Exostellar's Karpenter (xkarpenter)'s namespace."
  type        = string
  default     = "exokarpenter"
}

variable "pod_resource_limits" {
  description = "Pod resource requests and limits."
  type = map(
    object(
      {
        cpu    = number
        memory = string
      },
    ),
  )
  default = {
    requests = {
      cpu    = 1
      memory = "1Gi"
    }
    limits = {
      cpu    = 1
      memory = "1Gi"
    }
  }
}

variable "xkarpenter_version" {
  description = "Exostellar's Karpenter (xkarpenter)'s Helm chart's version. Note: To keep xKarpenter and xKarpenter Resources compatible, same version is used with both charts."
  type        = string
  default     = "v2.0.6"
}

# Optional: Permissions boundary for IAM roles created by IAM and EMS modules.
variable "permissions_boundary_arn" {
  description = "Optional ARN of an IAM policy to use as a permissions boundary for IAM roles created by the IAM and EMS modules."
  type        = string
  default     = null
}

variable "guest_kernel_version" {
  description = "Guest kernel version for xspot nodes."
  type        = string
  default     = "5.15.185"
}

##########################
### CNI and CSI Inputs ###
##########################

variable "exo_cni_chart_repository" {
  description = "Exostellar's CNI chart's repository URL."
  type        = string
  default     = "oci://public.ecr.aws/u8h5n6o4/exostellar-cni/exo-aws-vpc-cni"
}

variable "exo_cni_chart_version" {
  description = "Exostellar's CNI chart's version."
  type        = string
  default     = "1.20.0"
}

variable "exo_cni_chart_namespace" {
  description = "Exostellar's CNI chart's namespace."
  type        = string
  default     = "kube-system"
}

variable "exo_csi_chart_repository" {
  description = "Exostellar's CSI chart's repository URL."
  type        = string
  default     = "oci://public.ecr.aws/u8h5n6o4/exostellar-csi/exo-aws-ebs-csi-driver"
}

variable "exo_csi_chart_version" {
  description = "Exostellar's CSI chart's version."
  type        = string
  default     = "1.46.0"
}

variable "exo_csi_chart_namespace" {
  description = "Exostellar's CSI chart's namespace."
  type        = string
  default     = "kube-system"
}

############################
### Miscellaneous Inputs ###
############################

variable "aws_resource_prefix" {
  description = "Prefix to use for all AWS resources. Note: This will not add any trailing extra characters like '-'."
  type        = string
  default     = ""
}

variable "ssm_enabled" {
  description = "Enable AWS Systems Manager (SSM) for Exostellar Management Server (EMS)?"
  type        = bool
  default     = false
}
