locals {
  # Format the EKS node image's name from the EKS version. e.g., "1.32" -> "k8s-132".
  k8s_node_image_name = format("k8s-%s", replace(var.eks_version, ".", ""))
}

# Deploy EKS cluster and related resources.
#
# It performs the following actions:
# 1. Reads the available availability zones in the region.
# 2. Creates VPC and other networking resources for EKS cluster.
# 3. Creates EKS cluster with managed node group.
# 4. Creates IRSA setup for EBS CSI driver.
# 5. Reads the EKS cluster's auth info.
# 6. Reads the EKS cluster's TLS certificate details.
module "xio_infra" {
  source = "../infra"

  aws_region                  = var.aws_region
  eks_cluster                 = var.eks_cluster
  eks_version                 = var.eks_version
  vpc_cidr                    = var.vpc_cidr
  eks_node_group_desired_size = var.eks_node_group_desired_size
}

# Update kubeconfig to use the specified EKS cluster.
resource "null_resource" "update_kubeconfig" {
  depends_on = [module.xio_infra]

  provisioner "local-exec" {
    command = "aws eks update-kubeconfig --region ${var.aws_region} --name ${var.eks_cluster}"
  }
}

locals {
  # All (cluster + additional) security groups associated with the EKS cluster.
  cluster_security_groups = concat(
    [module.xio_infra.eks_cluster_security_group_id, module.xio_infra.eks_cluster_primary_security_group_id],
    module.xio_infra.eks_cluster_additional_security_group_ids,
  )
}

# Deploy the Exostellar's IAM resources.
#
# It performs the following actions:
# 1. Reads the IAM policy document for EC2 instance assume role.
# 2. Reads managed IAM policy AmazonEC2ContainerRegistryReadOnly.
# 3. Reads managed IAM policy AmazonSSMManagedInstanceCore.
# 4. Reads managed IAM policy AmazonEKS_CNI_Policy.
# 5. Reads managed IAM policy AmazonEKSWorkerNodePolicy.
# 6. Reads managed IAM policy AmazonEBSCSIDriverPolicy.
# 7. Creates IAM Role for xspot controller.
# 8. Creates IAM Policy for xspot controller.
# 9. Creates IAM Instance Profile for xspot controller.
# 10. Creates IAM Role for xspot worker.
# 11. Creates IAM Policy for xspot worker.
# 12. Creates IAM Instance Profile for xspot worker.
# 13. Attaches managed IAM policy AmazonEC2ContainerRegistryReadOnly to xspot worker.
# 14. Attaches managed IAM policy AmazonSSMManagedInstanceCore to xspot worker.
# 15. Attaches managed IAM policy AmazonEKS_CNI_Policy to xspot worker.
# 16. Attaches managed IAM policy AmazonEKSWorkerNodePolicy to xspot worker.
# 17. Attaches managed IAM policy AmazonEBSCSIDriverPolicy to xspot worker.
module "exostellar_iam" {
  source     = "../iam"
  depends_on = [module.xio_infra, null_resource.update_kubeconfig]

  aws_region               = var.aws_region
  eks_cluster              = var.eks_cluster
  permissions_boundary_arn = var.permissions_boundary_arn
  aws_resource_prefix      = var.aws_resource_prefix
}

# Deploy the Exostellar Management Server (EMS) and related resources.
#
# It performs the following actions:
# 1. Reads the subnet based on its ID.
# 2. Reads the IAM policy document for EC2 instance assume role.
# 3. Reads the AWS-managed IAM policy for SSM.
# 4. Creates a security Group for Exostellar Management Server (EMS).
# 5. Creates IAM Role for Exostellar Management Server (EMS).
# 6. Creates IAM Policy for Exostellar Management Server (EMS).
# 7. Attaches the SSM policy to the EMS role, only if SSM flag is enabled.
# 8. Creates IAM Instance Profile for Exostellar Management Server (EMS).
# 9. Creates the Exostellar Management Server (EMS) EC2 instance.
module "exostellar_management_server" {
  source     = "../ems"
  depends_on = [module.xio_infra, null_resource.update_kubeconfig, module.exostellar_iam]

  # Use common values like cluster name and AWS region from IAM module's outputs so EMS waits for IAM module's
  # deployment.
  eks_cluster = var.eks_cluster
  aws_region  = var.aws_region
  vpc_id      = module.xio_infra.vpc_id
  # Pass the CIDR block of VPC attached to the EKS cluster created.
  vpc_cidr_block   = var.vpc_cidr
  subnet_id        = module.xio_infra.public_subnets[0]
  is_subnet_public = true # By the design of "infra" module, the subnet ID provided will be public.
  # Pass all (cluster + additional) security groups IDs associated with the existing EKS cluster.
  shared_security_group_ids             = local.cluster_security_groups
  xspot_controller_subnet_id            = module.xio_infra.private_subnets[0]
  xspot_controller_instance_profile_arn = module.exostellar_iam.xspot_controller_instance_profile_arn
  xspot_worker_subnet_id                = module.xio_infra.private_subnets[0]
  xspot_worker_instance_profile_arn     = module.exostellar_iam.xspot_worker_instance_profile_arn
  k8s_node_image_name                   = local.k8s_node_image_name

  ssh_key_name = var.ssh_key_name
  ssm_enabled  = var.ssm_enabled
  region_ami_map = {
    (var.aws_region) : var.ems_ami_id
  }
  termination_protection       = var.ems_termination_protection
  volume_delete_on_termination = var.ems_volume_delete_on_termination
  encrypt_volume               = var.ems_volume_encryption
  volume_type                  = var.ems_volume_type
  xspot_enable_hyperthreading  = var.xspot_enable_hyperthreading
  xspot_enable_balloon         = var.xspot_enable_balloon
  profile_az                   = var.profile_availability_zone
  domain_name                  = var.domain_name
  instance_type                = var.ems_instance_type
  volume_size                  = var.ems_volume_size_gb
  nfs_dns_name                 = var.nfs_dns_name
  nfs_security_group_id        = var.nfs_security_group_id
  permissions_boundary_arn     = var.permissions_boundary_arn
  aws_resource_prefix          = var.aws_resource_prefix
}

# Configure the EKS cluster according to Exostellar’s standard setup.
#
# It performs the following actions:
# 1. Adds EKS access entry for xspot controller.
# 2. Associates EKS admin view policy with xspot controller.
# 3. Adds EKS access entry for xspot worker.
# 4. Associates EKS admin policy with xspot worker.
# 5. Creates OIDC provider for the EKS cluster.
# 6. Deploys Exostellar CNI's Helm chart.
# 7. Deploys Exostellar CSI's Helm chart.
module "eksconfig" {
  source = "../eksconfig"
  depends_on = [
    module.xio_infra,
    null_resource.update_kubeconfig,
    module.exostellar_iam,
  ]

  # EKS cluster inputs.
  eks_cluster                 = var.eks_cluster
  eks_cluster_oidc_issuer     = module.xio_infra.eks_cluster_oidc_issuer
  eks_cluster_oidc_thumbprint = module.xio_infra.eks_cluster_oidc_thumbprint


  # Xspot inputs.
  xspot_controller_role_arn = module.exostellar_iam.xspot_controller_role_arn
  xspot_worker_role_arn     = module.exostellar_iam.xspot_worker_role_arn

  # CNI and CSI inputs.
  exo_cni_chart_repository = var.exo_cni_chart_repository
  exo_cni_chart_version    = var.exo_cni_chart_version
  exo_cni_chart_namespace  = var.exo_cni_chart_namespace
  exo_csi_chart_repository = var.exo_csi_chart_repository
  exo_csi_chart_version    = var.exo_csi_chart_version
  exo_csi_chart_namespace  = var.exo_csi_chart_namespace
}

# Add license to Exostellar Management Server (EMS).
#
# It performs the following actions:
# 1. Waits for the x-compute API to be reachable.
# 2. Adds the license to EMS.
module "license" {
  source     = "../license"
  depends_on = [module.exostellar_management_server]
  # Skip running this module (i.e., adding license) if the license_filepath is not specified.
  count = var.license_filepath != "" ? 1 : 0

  ems_public_ip = module.exostellar_management_server.exostellar_management_server_public_ip
  # Read the license file and pass the data to the license module. Trim the trailing newline character.
  license_data = trim(file(var.license_filepath), "\n")
}

# Deploy Exostellar's Karpenter (xKarpenter) and related resources.
#
# It performs the following actions:
# 1. Creates Exostellar's Karpenter (xkarpenter) namespace
# 2. Creates IRSA for Exostellar's Karpenter (xkarpenter).
# 3. Attaches Karpenter policy to IAM role.
# 4. Creates IRSA for Exostellar's xnode-controller.
# 5. Attaches exo-node-controller policy to the IAM role.
# 6. Deploys the Exostellar's Karpenter (xkarpenter)'s Helm chart.
# 7. Deploys the Exostellar's Karpenter (xkarpenter) resources (default ExoNodeClass and ExoNodePool) Helm chart.
module "xkarpenter" {
  source = "../karpenter"
  depends_on = [
    module.xio_infra,
    null_resource.update_kubeconfig,
    module.exostellar_iam,
    module.exostellar_management_server,
  ]

  # EKS cluster inputs.
  eks_cluster                   = var.eks_cluster
  eks_cluster_oidc_issuer       = module.xio_infra.eks_cluster_oidc_issuer
  eks_cluster_oidc_provider_arn = module.eksconfig.eks_cluster_oidc_provider_arn

  # Use common values like cluster name from EMS module's outputs so xkarpenter waits for EMS module's deployment.
  exostellar_management_server_private_ip    = module.exostellar_management_server.exostellar_management_server_private_ip
  xkarpenter_helm_chart_repository           = var.xkarpenter_helm_chart_repository
  xkarpenter_resources_helm_chart_repository = var.xkarpenter_resources_helm_chart_repository
  namespace                                  = var.xkarpenter_namespace
  pod_resources                              = var.pod_resource_limits
  xkarpenter_version                         = var.xkarpenter_version
  xspot_controller_instance_profile_arn      = module.exostellar_iam.xspot_controller_instance_profile_arn
  xspot_worker_instance_profile_arn          = module.exostellar_iam.xspot_worker_instance_profile_arn
  region                                     = var.aws_region
  k8s_node_image_name                        = local.k8s_node_image_name
  # Pass the private subnet IDs associated with the existing EKS cluster. These will be used in the default
  # ExoNodeClass.
  eks_cluster_private_subnet_ids = module.xio_infra.private_subnets
  # Pass security group (cluster + additional) IDs associated with the existing EKS cluster. These will be attached to
  # the x-compute nodes.
  # 
  # If the flag allow_xspot_worker_inbound_traffic is enabled, include the x-spot security group in the list.
  xcompute_node_security_groups = concat(
    tolist(local.cluster_security_groups),
    # TODO :: Bhargav :: Add the following x-spot security group to the list of security groups after adding the xspot
    # module to standalone flow.
    # module.xspot[0].security_group_id != "" ? [module.xspot[0].security_group_id] : [],
  )
  # Pass the xspot version to set in the default ExoNodeClass.
  xspot_version = var.xspot_version
  # Enable Infrastructure Optimizer (IO) for xspot?
  enable_infrastructure_optimizer = var.enable_infrastructure_optimizer
  # Enable Workload Optimizer (WO) for xspot?
  enable_workload_optimizer = var.enable_workload_optimizer
  # Enable hyperthreading in Xspot? Default is true.
  xspot_enable_hyperthreading = var.xspot_enable_hyperthreading
  # Enable ballooning in Xspot? Default is true.
  xspot_enable_balloon = var.xspot_enable_balloon
  # Guest kernel version for xspot nodes. Default is "5.15.185".
  guest_kernel_version = var.guest_kernel_version
}
