locals {
  # Supported EKS versions.
  #
  # Note: This shouldn't be allowed to change by the caller. Hence, hardcoded.
  supported_eks_versions = ["1.29", "1.30", "1.31", "1.32"]

  # Supported EKS versions (as a string).
  supported_eks_versions_str = join(", ", local.supported_eks_versions)

  # EKS version validation error messages.
  eks_validation_error_messages = {
    "1.33"    = "❌ EKS version 1.33 is not yet supported, as AL2 AMIs for Kubernetes 1.33 are discontinued."
    "default" = "❌ EKS version ${var.eks_cluster_version} must be one of [${local.supported_eks_versions_str}]"
  }

  # Supported EKS authentication modes, i.e., only allow when API mode (EKS access entries) is enabled.
  #
  # Note: This shouldn't be allowed to change by the caller. Hence, hardcoded.
  supported_eks_auth_modes = ["API", "API_AND_CONFIG_MAP"]

  # Supported EKS authentication modes (as a string).
  supported_eks_auth_modes_str = join(", ", local.supported_eks_auth_modes)

  # EKS version validation error messages.
  eks_auth_validation_error_messages = {
    ""           = "❌ EKS auth mode cannot be empty."
    "CONFIG_MAP" = "❌ EKS auth mode CONFIG_MAP (aws-auth) is deprecated."
    # The default string is moved to an EOT block to avoid crossing the 120-character line limit.
    "default" = <<-EOT
      ❌ EKS auth mode ${var.eks_cluster_auth_mode} must be one of ${local.supported_eks_auth_modes_str}
    EOT
  }
}

# Validate that the EKS cluster version is within the supported range [1.29, 1.30, 1.31, 1.32].
resource "null_resource" "validate_eks_version" {
  lifecycle {
    precondition {
      condition = contains(local.supported_eks_versions, var.eks_cluster_version)
      error_message = (
        var.eks_cluster_version == "1.33" ? local.eks_validation_error_messages["1.33"] :
        local.eks_validation_error_messages["default"]
      )
    }
  }
}

# Validate that the EKS cluster’s authentication mode supports API access (EKS access entries).
resource "null_resource" "validate_eks_auth_mode" {
  lifecycle {
    precondition {
      condition = contains(local.supported_eks_auth_modes, var.eks_cluster_auth_mode)
      error_message = (
        var.eks_cluster_auth_mode == "" ? local.eks_auth_validation_error_messages[""] :
        var.eks_cluster_auth_mode == "CONFIG_MAP" ? local.eks_auth_validation_error_messages["CONFIG_MAP"] :
        local.eks_auth_validation_error_messages["default"]
      )
    }
  }
}
