variable "xkarpenter_helm_chart_repository" {
  description = "Exostellar's Karpenter (xkarpenter)'s Helm chart's repository URL."
  type        = string
  default     = "oci://public.ecr.aws/u8h5n6o4/exostellar-karpenter/karpenter"
}

variable "xkarpenter_resources_helm_chart_repository" {
  description = "Exostellar's Karpenter (xkarpenter)'s resources (ExoNodeClass and ExoNodePool) Helm chart's repository URL."
  type        = string
  default     = "oci://public.ecr.aws/u8h5n6o4/exostellar-karpenter/karpenter-resources"
}

variable "xkarpenter_version" {
  description = "Exostellar's Karpenter (xkarpenter)'s Helm chart's version. Note: To keep xKarpenter and xKarpenter Resources compatible, same version is used with both charts."
  type        = string
  default     = "v2.0.6"
}

variable "namespace" {
  description = "Exostellar's Karpenter (xkarpenter)'s namespace."
  type        = string
  default     = "exokarpenter"
}

variable "exostellar_management_server_private_ip" {
  description = "Exostellar management server (EMS)'s private IP address."
  type        = string
}

variable "eks_cluster" {
  description = "EKS cluster's name. It must be between 1 and 26 characters in length."
  type        = string

  # EKS cluster name length validation. It must be between 1 and 26 characters.
  validation {
    condition = (
      length(var.eks_cluster) > 0 &&
      length(var.eks_cluster) <= 26
    )
    error_message = <<EOT
EKS cluster name's length must be between 1 and 26 characters in the Exostellar modules. As the EKS cluster name will be
prefixed to all Exostellar resources, due to the length limitations on a few of the resources, the EKS cluster name must
be at most 26 characters long.
EOT
  }
}

variable "region" {
  description = "AWS region of cluster."
  default     = ""
  type        = string
}


variable "pod_resources" {
  description = "Pod resource requests and limits."
  type = map(
    object(
      {
        cpu    = number
        memory = string
      },
    ),
  )
  default = {
    requests = {
      cpu    = 1
      memory = "1Gi"
    }
    limits = {
      cpu    = 1
      memory = "1Gi"
    }
  }
}

variable "xspot_controller_instance_profile_arn" {
  description = "Xspot controller's instance-profile's ARN"
  type        = string
}

variable "xspot_worker_instance_profile_arn" {
  description = "Xspot worker's instance-profile's ARN"
  type        = string
}

variable "xspot_version" {
  description = "Xspot (controller and worker) version to use."
  type        = string
  # Example: "xspot-3.4.0".
  #
  # Note: Specifying a default value here might cause a mismatch between this and the user requested xspot version.
  # Hence, this is made mandatory for this module.
}

variable "k8s_node_image_name" {
  description = "Kubernetes node image name."
  type        = string
  # Example: "k8s-132".
  #
  # Note: Specifying a default value here might cause a mismatch between this and the actual EKS version. Hence, this is
  # made mandatory for this module.
}

variable "aws_resource_prefix" {
  description = "Prefix to use for all AWS resources. Note: This will not add any trailing extra characters like '-'."
  type        = string
  default     = ""
}

variable "eks_cluster_oidc_issuer" {
  description = "EKS cluster's OIDC issuer URL."
  type        = string
}

variable "eks_cluster_oidc_provider_arn" {
  description = "EKS cluster's OIDC provider ARN."
  type        = string
}

# List of private subnet IDs associated with the EKS cluster. These will be added to the default ExoNodeClass so that
# the xKarpenter can launch nodes in these subnets.
variable "eks_cluster_private_subnet_ids" {
  description = "List of private subnet IDs associated with the EKS cluster."
  type        = list(string)
}

# List of security groups to attach to x-compute nodes. These will be added to the default ExoNodeClass so that the
# xKarpenter can launch nodes with these security groups.
#
# The list is expected to contain the following security groups:
# 1. The EKS cluster's security group.
# 2. Additional security groups attached to the EKS cluster's control plane (if any).
# 3. (Optional) The xspot security group created using the `xspot` module for xspot workers to allow the following
#    access:
#    1. Allow TCP traffic from `0.0.0.0/0` for the ports ranging from `32768 - 60999`.
#    2. Allow all traffic within the VPC.
variable "xcompute_node_security_groups" {
  description = "List of security groups to attach to x-compute nodes."
  type        = list(string)
}

# Enable Infrastructure Optimizer (IO) for xspot? Defaults to true.
variable "enable_infrastructure_optimizer" {
  description = "Enable Infrastructure Optimizer (IO) for xspot?"
  type        = bool
  default     = true
}

# Enable Workload Optimizer (WO) for xspot? Defaults to true.
variable "enable_workload_optimizer" {
  description = "Enable Workload Optimizer (WO) for xspot?"
  type        = bool
  default     = true
}

# Enable hyperthreading in Xspot? Default is true.
variable "xspot_enable_hyperthreading" {
  description = "Enable hyperthreading in Xspot?"
  type        = bool
  default     = true
}

# Enable ballooning in Xspot? Default is true.
variable "xspot_enable_balloon" {
  description = "Enable ballooning in Xspot?"
  type        = bool
  default     = true
}

variable "guest_kernel_version" {
  description = "Guest kernel version for xspot nodes."
  type        = string
  default     = "5.15.185"
}
