locals {
  vpc_azs = tolist(
    compact(
      # Availability zone us-east-1e throws UnsupportedAvailabilityZoneException. Excluded if exists in list.
      [for az in data.aws_availability_zones.available.names : az if az != "us-east-1e"]
    )
  )

  module_tags = {
    cluster = var.eks_cluster
    region  = var.aws_region
  }
}

# Read the available availability zones in the region.
data "aws_availability_zones" "available" {
  state = "available"
}

# Create VPC and other networking resources for EKS cluster.
module "vpc" {
  source  = "terraform-aws-modules/vpc/aws"
  version = "6.0.1"

  name            = var.eks_cluster
  cidr            = var.vpc_cidr
  azs             = local.vpc_azs
  private_subnets = [for k, v in local.vpc_azs : cidrsubnet(var.vpc_cidr, 6, k)]
  public_subnets  = [for k, v in local.vpc_azs : cidrsubnet(var.vpc_cidr, 6, k + 6)]

  enable_dns_hostnames    = true
  enable_dns_support      = true
  enable_nat_gateway      = true
  single_nat_gateway      = true
  map_public_ip_on_launch = true

  tags                          = local.module_tags
  customer_gateway_tags         = local.module_tags
  database_acl_tags             = local.module_tags
  database_route_table_tags     = local.module_tags
  database_subnet_group_tags    = local.module_tags
  database_subnet_tags          = local.module_tags
  default_network_acl_tags      = local.module_tags
  default_route_table_tags      = local.module_tags
  default_security_group_tags   = local.module_tags
  default_vpc_tags              = local.module_tags
  dhcp_options_tags             = local.module_tags
  elasticache_acl_tags          = local.module_tags
  elasticache_route_table_tags  = local.module_tags
  elasticache_subnet_group_tags = local.module_tags
  elasticache_subnet_tags       = local.module_tags
  igw_tags                      = local.module_tags
  intra_acl_tags                = local.module_tags
  intra_route_table_tags        = local.module_tags
  intra_subnet_tags             = local.module_tags
  nat_eip_tags                  = local.module_tags
  nat_gateway_tags              = local.module_tags
  outpost_acl_tags              = local.module_tags
  outpost_subnet_tags           = local.module_tags
  private_acl_tags              = local.module_tags
  private_route_table_tags      = local.module_tags
  private_subnet_tags           = local.module_tags
  public_acl_tags               = local.module_tags
  public_route_table_tags       = local.module_tags
  public_subnet_tags            = local.module_tags
  redshift_acl_tags             = local.module_tags
  redshift_route_table_tags     = local.module_tags
  redshift_subnet_group_tags    = local.module_tags
  redshift_subnet_tags          = local.module_tags
  vpc_flow_log_tags             = local.module_tags
  vpc_tags                      = local.module_tags
  vpn_gateway_tags              = local.module_tags
}

# Create EKS cluster with managed node group.
module "eks" {
  source  = "terraform-aws-modules/eks/aws"
  version = "21.1.5"

  name                   = var.eks_cluster
  kubernetes_version     = var.eks_version
  endpoint_public_access = true

  addons = {
    coredns                = {}
    eks-pod-identity-agent = {}
    kube-proxy             = {}
    vpc-cni                = {}
    aws-ebs-csi-driver = {
      service_account_role_arn = module.ebs_csi_irsa_role.arn
    }
  }

  vpc_id                        = module.vpc.vpc_id
  subnet_ids                    = concat(module.vpc.public_subnets, module.vpc.private_subnets)
  additional_security_group_ids = [module.vpc.default_security_group_id]

  tags                      = local.module_tags
  node_iam_role_tags        = local.module_tags
  cloudwatch_log_group_tags = local.module_tags
  encryption_policy_tags    = local.module_tags
  security_group_tags       = local.module_tags
  cluster_tags              = local.module_tags
  iam_role_tags             = local.module_tags
  eks_managed_node_groups = {
    "${var.eks_cluster}" = {
      # Starting on 1.30, AL2023 is the default AMI type for EKS managed node groups
      ami_type                                     = "AL2023_x86_64_STANDARD"
      instance_types                               = ["m5.xlarge"]
      node_security_group_enable_recommended_rules = true

      min_size     = 1
      max_size     = 3
      desired_size = var.eks_node_group_desired_size

      tags = local.module_tags

      iam_role_additional_policies = {
        AmazonSSMManagedInstanceCore = "arn:aws:iam::aws:policy/AmazonSSMManagedInstanceCore"
        AmazonEBSCSIDriverPolicy     = "arn:aws:iam::aws:policy/service-role/AmazonEBSCSIDriverPolicy"
      }
    }
  }

  enable_cluster_creator_admin_permissions = true

  timeouts = {
    create = "30m"
    delete = "30m"
  }
}

# Create IRSA setup for EBS CSI driver.
module "ebs_csi_irsa_role" {
  # Source GitHub Repo:
  # github.com/terraform-aws-modules/terraform-aws-iam/blob/v6.2.1/modules/iam-role-for-service-accounts/README.md
  source  = "terraform-aws-modules/iam/aws//modules/iam-role-for-service-accounts"
  version = "6.2.1"

  # Name of the IAM Role to be created. This will be used for IRSA.
  name = "${var.eks_cluster}-ebs-csi"

  # Attach the custom EBS CSI driver policy to this IAM role. Refer to file
  # `modules/iam-role-for-service-accounts/policies.tf` in source GitHub repo for more details.
  attach_ebs_csi_policy = true

  # Tag the EKS cluster's OIDC provider and map the service account for IRSA auth.
  oidc_providers = {
    ex = {
      # The ARN of the OIDC identity provider associated with the EKS cluster.
      provider_arn = module.eks.oidc_provider_arn

      # EBS CSI driver service account name "ebs-csi-controller-sa" along with namespace "kube-system". This IAM role
      # will be assumed by that service account so that the EBS CSI driver can interact with AWS API to manage volumes,
      # etc.
      namespace_service_accounts = [
        "kube-system:ebs-csi-controller-sa"
      ]
    }
  }

  # Add common tags to the IAM role.
  tags = local.module_tags
}

# Read the EKS cluster's auth info.
data "aws_eks_cluster_auth" "eks_cluster_auth" {
  depends_on = [module.eks]

  name = var.eks_cluster
}

# Read the EKS cluster's TLS certificate details.
data "tls_certificate" "eks_cluster_oidc" {
  url = module.eks.cluster_oidc_issuer_url
}
