locals {
  module_tags = {
    cluster = var.eks_cluster
    region  = var.aws_region
  }
}

# Read the IAM policy document for EC2 instance assume role.
data "aws_iam_policy_document" "instance_assume_role_policy" {
  statement {
    actions = ["sts:AssumeRole"]
    principals {
      type        = "Service"
      identifiers = ["ec2.amazonaws.com"]
    }
  }
}

# Read managed IAM policy AmazonEC2ContainerRegistryReadOnly.
data "aws_iam_policy" "managed_ec2_container_registry_read_only_policy" {
  arn = "arn:aws:iam::aws:policy/AmazonEC2ContainerRegistryReadOnly"
}

# Read managed IAM policy AmazonSSMManagedInstanceCore.
data "aws_iam_policy" "managed_amazon_ssm_managed_instance_core_policy" {
  arn = "arn:aws:iam::aws:policy/AmazonSSMManagedInstanceCore"
}

# Read managed IAM policy AmazonEKS_CNI_Policy.
data "aws_iam_policy" "managed_eks_cni_policy" {
  arn = "arn:aws:iam::aws:policy/AmazonEKS_CNI_Policy"
}

# Read managed IAM policy AmazonEKSWorkerNodePolicy.
data "aws_iam_policy" "managed_eks_worker_node_policy" {
  arn = "arn:aws:iam::aws:policy/AmazonEKSWorkerNodePolicy"
}

# Read managed IAM policy AmazonEBSCSIDriverPolicy.
data "aws_iam_policy" "managed_ebs_csi_driver_policy" {
  arn = "arn:aws:iam::aws:policy/service-role/AmazonEBSCSIDriverPolicy"
}

# Create IAM Role for xspot controller.
resource "aws_iam_role" "xspot_controller_role" {
  name                 = "${var.aws_resource_prefix}${var.eks_cluster}-xspot-controller-role"
  assume_role_policy   = data.aws_iam_policy_document.instance_assume_role_policy.json
  permissions_boundary = var.permissions_boundary_arn
  tags                 = local.module_tags
}

# Create IAM Policy for xspot controller.
resource "aws_iam_role_policy" "xspot_controller_role_policy" {
  name   = "${var.aws_resource_prefix}${var.eks_cluster}-xspot-controller-policy"
  role   = aws_iam_role.xspot_controller_role.id
  policy = file("${path.module}/policy/xspot-controller.json")
}

# Create IAM Instance Profile for xspot controller.
resource "aws_iam_instance_profile" "xspot_controller_instance_profile" {
  name = "${var.aws_resource_prefix}${var.eks_cluster}-xspot-controller-profile"
  role = aws_iam_role.xspot_controller_role.name
  tags = local.module_tags
}

# Create IAM Role for xspot worker.
resource "aws_iam_role" "xspot_worker_role" {
  name                 = "${var.aws_resource_prefix}${var.eks_cluster}-xspot-worker-role"
  assume_role_policy   = data.aws_iam_policy_document.instance_assume_role_policy.json
  permissions_boundary = var.permissions_boundary_arn
  tags                 = local.module_tags
}

# Create IAM Policy for xspot worker.
resource "aws_iam_role_policy" "xspot_worker_role_policy" {
  name   = "${var.aws_resource_prefix}${var.eks_cluster}-xspot-worker-policy"
  role   = aws_iam_role.xspot_worker_role.id
  policy = file("${path.module}/policy/xspot-worker.json")
}

# Create IAM Instance Profile for xspot worker.
resource "aws_iam_instance_profile" "xspot_worker_instance_profile" {
  name = "${var.aws_resource_prefix}${var.eks_cluster}-xspot-worker-profile"
  role = aws_iam_role.xspot_worker_role.name
  tags = local.module_tags
}

# Attach managed IAM policy AmazonEC2ContainerRegistryReadOnly to xspot worker.
resource "aws_iam_role_policy_attachment" "xspot_worker_role_managed_ec2_container_registry_read_only" {
  role       = aws_iam_role.xspot_worker_role.name
  policy_arn = data.aws_iam_policy.managed_ec2_container_registry_read_only_policy.arn
}

# Attach managed IAM policy AmazonSSMManagedInstanceCore to xspot worker.
resource "aws_iam_role_policy_attachment" "xspot_worker_role_managed_amazon_ssm_managed_instance_core" {
  role       = aws_iam_role.xspot_worker_role.name
  policy_arn = data.aws_iam_policy.managed_amazon_ssm_managed_instance_core_policy.arn
}

# Attach managed IAM policy AmazonEKS_CNI_Policy to xspot worker.
resource "aws_iam_role_policy_attachment" "xspot_worker_role_managed_eks_cni" {
  role       = aws_iam_role.xspot_worker_role.name
  policy_arn = data.aws_iam_policy.managed_eks_cni_policy.arn
}

# Attach managed IAM policy AmazonEKSWorkerNodePolicy to xspot worker.
resource "aws_iam_role_policy_attachment" "xspot_worker_role_managed_eks_worker_node" {
  role       = aws_iam_role.xspot_worker_role.name
  policy_arn = data.aws_iam_policy.managed_eks_worker_node_policy.arn
}

# Attach managed IAM policy AmazonEBSCSIDriverPolicy to xspot worker.
resource "aws_iam_role_policy_attachment" "xspot_worker_role_managed_ebs_csi_driver" {
  role       = aws_iam_role.xspot_worker_role.name
  policy_arn = data.aws_iam_policy.managed_ebs_csi_driver_policy.arn
}
