locals {
  # Common tags to apply to all resources in this module.
  module_tags = {
    cluster = var.eks_cluster
    region  = var.aws_region
  }
}

# Create an IAM Role for Service Account (IRSA) for the Amazon EBS CSI driver.
module "ebs_csi_driver_irsa_role" {
  # Source GitHub Repo:
  # github.com/terraform-aws-modules/terraform-aws-iam/blob/v6.2.1/modules/iam-role-for-service-accounts/README.md
  source  = "terraform-aws-modules/iam/aws//modules/iam-role-for-service-accounts"
  version = "6.2.1"

  # Name of the IAM Role to be created. This will be used for IRSA.
  name = "${var.eks_cluster}-ebs-csi"
  # Name of the custom EBS CSI driver IAM policy.
  policy_name = "${var.eks_cluster}-ebs-csi"
  # Flag `use_name_prefix`, enabled by default, will use the above-mentioned name as a prefix to the IAM role. In other
  # words, the role name will be suffixed with a random numeric string as follows:
  # Example: "<cluster-name>-ebs-csi-20250916144511890600000002"
  #
  # Since the EBS CSI driver requires annotating the *exact* IAM role ARN in the EBS CSI's Kubernetes service account,
  # the flag `use_name_prefix` is disabled, so it will be easier for users to annotate.
  # 
  # Note: Annotating the EBS CSI driver service account with the exact IAM role ARN is a manual step in case of
  # existing-cluster-flow.
  use_name_prefix = false

  # Attach the custom EBS CSI driver policy to this IAM role. Refer to file
  # `modules/iam-role-for-service-accounts/policies.tf` in source GitHub repo for more details.
  attach_ebs_csi_policy = true

  # Tag the EKS cluster's OIDC provider and map the service account for IRSA auth.
  oidc_providers = {
    ex = {
      # The ARN of the OIDC identity provider associated with the EKS cluster.
      provider_arn = var.eks_cluster_oidc_provider_arn

      # EBS CSI driver service account name "ebs-csi-controller-sa" along with namespace "kube-system". This IAM role
      # will be assumed by that service account so that the EBS CSI driver can interact with AWS API to manage volumes,
      # etc.
      namespace_service_accounts = [
        "kube-system:ebs-csi-controller-sa"
      ]
    }
  }

  # Add common tags to the IAM role.
  tags = local.module_tags
}
