locals {
  # Format the EKS node image's name from the EKS version. e.g., "1.32" -> "k8s-132".
  k8s_node_image_name = format("k8s-%s", replace(data.aws_eks_cluster.eks_cluster.version, ".", ""))
}

# Read the EKS cluster's state.
data "aws_eks_cluster" "eks_cluster" {
  name = var.eks_cluster
}

# Read the EKS cluster's auth info.
data "aws_eks_cluster_auth" "eks_cluster_auth" {
  name = var.eks_cluster
}

# Read the VPC details.
data "aws_vpc" "cluster_vpc" {
  id = local.vpc_id
}

# Read the EKS cluster's TLS certificate details.
data "tls_certificate" "eks_oidc" {
  url = data.aws_eks_cluster.eks_cluster.identity[0].oidc[0].issuer
}

# Read the subnets (both public + private) in the VPC.
data "aws_subnets" "all" {
  filter {
    name   = "vpc-id"
    values = [local.vpc_id]
  }
}

# Read each subnet's details.
#
# Note: This is required to determine whether the subnet is public or private.
data "aws_subnet" "details" {
  for_each = toset(data.aws_subnets.all.ids)
  id       = each.value
}

# Read the route table's details for subnets (both public + private).
#
# Note: This helps identify public subnets (those with routes to an Internet Gateway).
data "aws_route_table" "by_subnet" {
  for_each  = toset(data.aws_subnets.all.ids)
  subnet_id = each.value
}

locals {
  # VPC ID associated with the EKS cluster.
  vpc_id = data.aws_eks_cluster.eks_cluster.vpc_config[0].vpc_id

  # Public subnet IDs based on the internet gateway ID associated with them in the route tables.
  public_subnet_ids = tolist(
    toset(
      flatten(
        [
          for rt in values(data.aws_route_table.by_subnet) : [
            for assoc in rt.associations : (
              anytrue([for r in rt.routes : startswith(r.gateway_id, "igw-")]) && assoc.subnet_id != null
              ? [assoc.subnet_id]
              : []
            )
          ]
        ]
      )
    )
  )

  # Private subnet IDs by excluding public subnets from all subnets in the VPC.
  private_subnet_ids = tolist(
    toset(
      [
        for id in data.aws_subnets.all.ids : id
        if !contains(local.public_subnet_ids, id)
      ]
    )
  )

  # Map subnet ID → availability zone
  subnet_to_availability_zone_map = {
    for id in data.aws_subnets.all.ids : id => data.aws_subnet.details[id].availability_zone
  }

  # Select alphabetically first availability zone.
  selected_availability_zone = try(
    sort(
      toset(
        [
          for id in data.aws_subnets.all.ids : data.aws_subnet.details[id].availability_zone
        ]
      )
    )[0]
  )

  # Public subnet in the selected availability zone.
  selected_public_subnet = try(
    [
      for id in local.public_subnet_ids : id
      if local.subnet_to_availability_zone_map[id] == local.selected_availability_zone
    ][0],
    null
  )

  # Private subnet in the selected availability zone.
  selected_private_subnet = try(
    [
      for id in local.private_subnet_ids : id
      if local.subnet_to_availability_zone_map[id] == local.selected_availability_zone
    ][0],
    null
  )

  # Cluster security group for the EKS cluster.
  cluster_security_group = data.aws_eks_cluster.eks_cluster.vpc_config[0].cluster_security_group_id

  # Additional security groups for the EKS cluster.
  additional_security_groups = data.aws_eks_cluster.eks_cluster.vpc_config[0].security_group_ids

  # All (cluster + additional) security groups associated with the EKS cluster.
  cluster_security_groups = toset(
    concat(
      [local.cluster_security_group],
      tolist(local.additional_security_groups)
    )
  )
}

# Update kubeconfig to use the specified EKS cluster.
resource "null_resource" "update_kubeconfig" {
  provisioner "local-exec" {
    command = "aws eks update-kubeconfig --region ${var.aws_region} --name ${var.eks_cluster}"
  }
}

# Perform prechecks.
#
# It performs the following actions:
# 1. Validates that the EKS cluster version is within the supported range [1.29, 1.30, 1.31, 1.32].
# 2. Validates that the EKS cluster’s authentication mode supports API access (EKS access entries).
module "precheck" {
  # Use the relative path to the precheck module's directory.
  # 
  # In case of importing precheck module separately, i.e., without the existing-cluster-full module, use the latest
  # precheck module release as source.
  # source = "git::ssh://git@github.com/Exostellar/terraform-exostellar-modules//modules/precheck?ref=v0.0.5"
  # 
  source = "../precheck"

  depends_on = [null_resource.update_kubeconfig]

  eks_cluster_version   = data.aws_eks_cluster.eks_cluster.version
  eks_cluster_auth_mode = data.aws_eks_cluster.eks_cluster.access_config[0].authentication_mode
}

# Deploy the Exostellar's IAM resources.
#
# It performs the following actions:
# 1. Reads the IAM policy document for EC2 instance assume role.
# 2. Reads managed IAM policy AmazonEC2ContainerRegistryReadOnly.
# 3. Reads managed IAM policy AmazonSSMManagedInstanceCore.
# 4. Reads managed IAM policy AmazonEKS_CNI_Policy.
# 5. Reads managed IAM policy AmazonEKSWorkerNodePolicy.
# 6. Reads managed IAM policy AmazonEBSCSIDriverPolicy.
# 7. Creates IAM Role for xspot controller.
# 8. Creates IAM Policy for xspot controller.
# 9. Creates IAM Instance Profile for xspot controller.
# 10. Creates IAM Role for xspot worker.
# 11. Creates IAM Policy for xspot worker.
# 12. Creates IAM Instance Profile for xspot worker.
# 13. Attaches managed IAM policy AmazonEC2ContainerRegistryReadOnly to xspot worker.
# 14. Attaches managed IAM policy AmazonSSMManagedInstanceCore to xspot worker.
# 15. Attaches managed IAM policy AmazonEKS_CNI_Policy to xspot worker.
# 16. Attaches managed IAM policy AmazonEKSWorkerNodePolicy to xspot worker.
# 17. Attaches managed IAM policy AmazonEBSCSIDriverPolicy to xspot worker.
module "exostellar_iam" {
  # Use the relative path to the iam module's directory.
  # 
  # In case of importing iam module separately, i.e., without the existing-cluster-full module, use the latest iam
  # module release as source.
  # source = "git::ssh://git@github.com/Exostellar/terraform-exostellar-modules//modules/iam?ref=v0.0.5"
  # 
  source = "../iam"

  depends_on = [module.precheck]

  aws_region               = var.aws_region
  eks_cluster              = var.eks_cluster
  aws_resource_prefix      = var.aws_resource_prefix
  permissions_boundary_arn = var.permissions_boundary_arn
}

# Deploy the xspot security group.
#
# It performs the following actions:
# 1. Create security group for xspot (controllers and workers).
module "xspot" {
  # Use the relative path to the xspot module's directory.
  # 
  # In case of importing xspot module separately, i.e., without the existing-cluster-full module, use the latest xspot
  # module release as source.
  # source = "git::ssh://git@github.com/Exostellar/terraform-exostellar-modules//modules/xspot?ref=v0.0.5"
  # 
  source = "../xspot"

  count      = var.allow_xspot_worker_inbound_traffic ? 1 : 0
  depends_on = [module.precheck]

  aws_region          = var.aws_region
  eks_cluster         = var.eks_cluster
  vpc_id              = local.vpc_id
  vpc_cidr_block      = data.aws_vpc.cluster_vpc.cidr_block
  aws_resource_prefix = var.aws_resource_prefix
}

# Deploy the Exostellar Management Server (EMS) and related resources.
#
# It performs the following actions:
# 1. Reads the subnet based on its ID.
# 2. Reads the IAM policy document for EC2 instance assume role.
# 3. Reads the AWS-managed IAM policy for SSM.
# 4. Creates a security Group for Exostellar Management Server (EMS).
# 5. Creates IAM Role for Exostellar Management Server (EMS).
# 6. Creates IAM Policy for Exostellar Management Server (EMS).
# 7. Attaches the SSM policy to the EMS role, only if SSM flag is enabled.
# 8. Creates IAM Instance Profile for Exostellar Management Server (EMS).
# 9. Creates the Exostellar Management Server (EMS) EC2 instance.
module "exostellar_management_server" {
  # Use the relative path to the ems module's directory.
  # 
  # In case of importing ems module separately, i.e., without the existing-cluster-full module, use the latest ems
  # module release as source.
  # source = "git::ssh://git@github.com/Exostellar/terraform-exostellar-modules//modules/ems?ref=v0.0.5"
  # 
  source = "../ems"

  depends_on = [null_resource.update_kubeconfig, module.exostellar_iam, module.xspot]

  # Use common values like cluster name and AWS region from IAM module's outputs so EMS waits for IAM module's
  # deployment.
  eks_cluster = var.eks_cluster
  aws_region  = var.aws_region
  vpc_id      = local.vpc_id
  # Pass the CIDR block of VPC attached to the existing EKS cluster.
  vpc_cidr_block = data.aws_vpc.cluster_vpc.cidr_block
  subnet_id      = local.selected_public_subnet
  # By the current design, the subnet for EMS must be public.
  is_subnet_public = true
  # Pass all (cluster + additional) security groups IDs associated with the existing EKS cluster.
  shared_security_group_ids = local.cluster_security_groups
  # Set xspot_security_group_id only if allow_xspot_worker_inbound_traffic is enabled. Else, set to an empty string.
  #
  # Since count is used in module.xspot, Terraform considers that as a list(tuple). Hence, use index module.xspot[0].
  xspot_security_group_id = (
    var.allow_xspot_worker_inbound_traffic
    ? module.xspot[0].security_group_id
    : ""
  )
  xspot_controller_subnet_id            = local.selected_private_subnet
  xspot_controller_instance_profile_arn = module.exostellar_iam.xspot_controller_instance_profile_arn
  xspot_worker_subnet_id                = local.selected_private_subnet
  xspot_worker_instance_profile_arn     = module.exostellar_iam.xspot_worker_instance_profile_arn
  k8s_node_image_name                   = local.k8s_node_image_name

  ssh_key_name = var.ssh_key_name
  region_ami_map = {
    (var.aws_region) : var.ems_ami_id
  }
  termination_protection       = var.ems_termination_protection
  xspot_enable_hyperthreading  = var.xspot_enable_hyperthreading
  xspot_enable_balloon         = var.xspot_enable_balloon
  profile_az                   = local.selected_availability_zone
  domain_name                  = var.domain_name
  instance_type                = var.ems_instance_type
  volume_size                  = var.ems_volume_size_gb
  volume_delete_on_termination = var.ems_volume_delete_on_termination
  encrypt_volume               = var.ems_volume_encryption
  volume_type                  = var.ems_volume_type
  nfs_dns_name                 = var.nfs_dns_name
  nfs_security_group_id        = var.nfs_security_group_id
  aws_resource_prefix          = var.aws_resource_prefix
  ssm_enabled                  = var.ssm_enabled
  permissions_boundary_arn     = var.permissions_boundary_arn
}

# Configure the EKS cluster according to Exostellar’s standard setup.
#
# It performs the following actions:
# 1. Adds EKS access entry for xspot controller.
# 2. Associates EKS admin view policy with xspot controller.
# 3. Adds EKS access entry for xspot worker.
# 4. Associates EKS admin policy with xspot worker.
# 5. Creates OIDC provider for the EKS cluster.
# 6. Deploys Exostellar CNI's Helm chart.
# 7. Deploys Exostellar CSI's Helm chart.
module "eksconfig" {
  # Use the relative path to the eksconfig module's directory.
  # 
  # In case of importing eksconfig module separately, i.e., without the existing-cluster-full module, use the latest
  # eksconfig module release as source.
  # source = "git::ssh://git@github.com/Exostellar/terraform-exostellar-modules//modules/eksconfig?ref=v0.0.5"
  # 
  source = "../eksconfig"

  depends_on = [
    null_resource.update_kubeconfig,
    module.exostellar_iam,
  ]

  # EKS cluster inputs.
  eks_cluster                 = var.eks_cluster
  eks_cluster_oidc_issuer     = data.aws_eks_cluster.eks_cluster.identity[0].oidc[0].issuer
  eks_cluster_oidc_thumbprint = data.tls_certificate.eks_oidc.certificates[0].sha1_fingerprint

  # Xspot inputs.
  xspot_controller_role_arn = module.exostellar_iam.xspot_controller_role_arn
  xspot_worker_role_arn     = module.exostellar_iam.xspot_worker_role_arn

  # CNI and CSI inputs.
  exo_cni_chart_repository = var.exo_cni_chart_repository
  exo_cni_chart_version    = var.exo_cni_chart_version
  exo_cni_chart_namespace  = var.exo_cni_chart_namespace
  exo_csi_chart_repository = var.exo_csi_chart_repository
  exo_csi_chart_version    = var.exo_csi_chart_version
  exo_csi_chart_namespace  = var.exo_csi_chart_namespace
}

# Deploy IRSA for the EBS CSI driver.
#
# It performs the following actions:
# 1. Create an IAM Role for Service Account (IRSA) for the Amazon EBS CSI driver.
module "iam_ebs_csi_driver_irsa" {
  source = "../iam-ebs-csi-driver-irsa"

  aws_region                    = var.aws_region
  eks_cluster                   = var.eks_cluster
  eks_cluster_oidc_provider_arn = module.eksconfig.eks_cluster_oidc_provider_arn
}

# Add license to Exostellar Management Server (EMS).
#
# It performs the following actions:
# 1. Waits for the x-compute API to be reachable.
# 2. Adds the license to EMS.
module "license" {
  # Use the relative path to the license module's directory.
  # 
  # In case of importing license module separately, i.e., without the existing-cluster-full module, use the latest
  # license module release as source.
  # source = "git::ssh://git@github.com/Exostellar/terraform-exostellar-modules//modules/license?ref=v0.0.5"
  # 
  source = "../license"

  depends_on = [module.exostellar_management_server]
  # Skip running this module (i.e., adding license) if the license_filepath is not specified.
  count = var.license_filepath != "" ? 1 : 0

  ems_public_ip = module.exostellar_management_server.exostellar_management_server_public_ip
  # Read the license file and pass the data to the license module. Trim the trailing newline character.
  license_data = trim(file(var.license_filepath), "\n")
}

# Deploy Exostellar's Karpenter (xKarpenter) and related resources.
#
# It performs the following actions:
# 1. Creates Exostellar's Karpenter (xkarpenter) namespace
# 2. Creates IRSA for Exostellar's Karpenter (xkarpenter).
# 3. Attaches Karpenter policy to IAM role.
# 4. Creates IRSA for Exostellar's xnode-controller.
# 5. Attaches exo-node-controller policy to the IAM role.
# 6. Deploys the Exostellar's Karpenter (xkarpenter)'s Helm chart.
# 7. Deploys the Exostellar's Karpenter (xkarpenter) resources (default ExoNodeClass and ExoNodePool) Helm chart.
module "xkarpenter" {
  # Use the relative path to the karpenter module's directory.
  # 
  # In case of importing karpenter module separately, i.e., without the existing-cluster-full module, use the latest
  # karpenter module release as source.
  # source = "git::ssh://git@github.com/Exostellar/terraform-exostellar-modules//modules/karpenter?ref=v0.0.5"
  # 
  source = "../karpenter"

  depends_on = [null_resource.update_kubeconfig, module.exostellar_iam, module.exostellar_management_server]

  # Use common values like cluster name from EMS module's outputs so xkarpenter waits for EMS module's deployment.
  eks_cluster                                = var.eks_cluster
  eks_cluster_oidc_issuer                    = data.aws_eks_cluster.eks_cluster.identity[0].oidc[0].issuer
  eks_cluster_oidc_provider_arn              = module.eksconfig.eks_cluster_oidc_provider_arn
  exostellar_management_server_private_ip    = module.exostellar_management_server.exostellar_management_server_private_ip
  xkarpenter_helm_chart_repository           = var.xkarpenter_helm_chart_repository
  xkarpenter_resources_helm_chart_repository = var.xkarpenter_resources_helm_chart_repository
  namespace                                  = var.xkarpenter_namespace
  pod_resources                              = var.pod_resource_limits
  xkarpenter_version                         = var.xkarpenter_version
  xspot_controller_instance_profile_arn      = module.exostellar_iam.xspot_controller_instance_profile_arn
  xspot_worker_instance_profile_arn          = module.exostellar_iam.xspot_worker_instance_profile_arn
  region                                     = var.aws_region
  k8s_node_image_name                        = local.k8s_node_image_name
  # Pass the private subnet IDs associated with the existing EKS cluster. These will be used in the default
  # ExoNodeClass.
  eks_cluster_private_subnet_ids = local.private_subnet_ids
  # Pass security group (cluster + additional) IDs associated with the existing EKS cluster. These will be attached to
  # the x-compute nodes.
  # 
  # If the flag allow_xspot_worker_inbound_traffic is enabled, include the x-spot security group in the list.
  xcompute_node_security_groups = concat(
    tolist(local.cluster_security_groups),
    module.xspot[0].security_group_id != "" ? [module.xspot[0].security_group_id] : [],
  )
  # Pass the xspot version to set in the default ExoNodeClass.
  xspot_version = var.xspot_version
  # Enable Infrastructure Optimizer (IO) for xspot?
  enable_infrastructure_optimizer = var.enable_infrastructure_optimizer
  # Enable Workload Optimizer (WO) for xspot?
  enable_workload_optimizer = var.enable_workload_optimizer
  # Enable hyperthreading in Xspot? Default is true.
  xspot_enable_hyperthreading = var.xspot_enable_hyperthreading
  # Enable ballooning in Xspot? Default is true.
  xspot_enable_balloon = var.xspot_enable_balloon
  # Guest kernel version for xspot nodes. Default is "5.15.185".
  guest_kernel_version = var.guest_kernel_version
}
