variable "aws_region" {
  description = "AWS region for deploying resources. Eg. 'us-east-1'."
  type        = string
}

variable "eks_cluster" {
  description = "EKS cluster's name. It must be between 1 and 26 characters in length."
  type        = string

  # EKS cluster name length validation. It must be between 1 and 26 characters.
  validation {
    condition = (
      length(var.eks_cluster) > 0 &&
      length(var.eks_cluster) <= 26
    )
    error_message = <<EOT
EKS cluster name's length must be between 1 and 26 characters in the Exostellar modules. As the EKS cluster name will be
prefixed to all Exostellar resources, due to the length limitations on a few of the resources, the EKS cluster name must
be at most 26 characters long.
EOT
  }
}

variable "vpc_id" {
  description = "EKS cluster's VPC ID."
  type        = string
}

variable "vpc_cidr_block" {
  description = "EKS cluster's VPC CIDR block."
  type        = string
}

variable "subnet_id" {
  description = "EKS cluster's subnet ID."
  type        = string
}

variable "is_subnet_public" {
  description = "Is EKS cluster's subnet public?"
  type        = bool
  default     = true
}

variable "domain_name" {
  description = "Domain name to run Exostellar Management Server (EMS) in. Optional."
  type        = string
  default     = ""
}

variable "shared_security_group_ids" {
  description = "Share security groups' IDs to attach to Exostellar Management Server (EMS). Optional."
  type        = list(string)
  default     = []
}

variable "instance_type" {
  description = "EC2 instance type for Exostellar Management Server (EMS)."
  type        = string
  default     = "m5d.xlarge"
}

variable "volume_size" {
  description = "Exostellar Management Server (EMS) instance's root volume size (in GB)."
  type        = number
  default     = 100
}

variable "volume_delete_on_termination" {
  description = "Delete the Exostellar Management Server (EMS) instance's root volume on termination of instance?"
  type        = bool
  default     = true
}

variable "encrypt_volume" {
  description = "Encrypt the Exostellar Management Server (EMS) instance's root volume?"
  type        = bool
  default     = false
}

variable "volume_type" {
  description = "Exostellar Management Server (EMS) instance's root volume type. Valid options: standard, gp2, gp3, io1, io2, sc1, and st1."
  type        = string
  default     = "gp2"

  validation {
    condition     = contains(["standard", "gp2", "gp3", "io1", "io2", "sc1", "st1"], var.volume_type)
    error_message = "Volume type must be one of: standard, gp2, gp3, io1, io2, sc1, st1."
  }
}

variable "ssh_key_name" {
  description = "SSH key-pair's name in AWS for Exostellar Management Server (EMS)'s access. If ssm_enabled is set to true, this will be skipped."
  type        = string
  default     = ""
}

variable "termination_protection" {
  description = "Enable termination protection for Exostellar Management Server (EMS)'s EC2 instance?"
  type        = bool
  default     = true
}

variable "region_ami_map" {
  description = "Mapping of AWS region to Exostellar Management Server (EMS)'s AMI."
  type        = map(string)
}

variable "nfs_dns_name" {
  description = "DNS name of the NFS file system for Exostellar Management Server (EMS)."
  type        = string
  default     = ""
}

variable "nfs_security_group_id" {
  description = "Security group ID that enable traffic between the NFS server and Exostellar Management Server (EMS)."
  type        = string
  default     = ""
}

variable "xspot_controller_subnet_id" {
  description = "Xspot controller's subnet ID."
  type        = string
}

variable "xspot_worker_subnet_id" {
  description = "Xspot worker's subnet ID."
  type        = string
}

variable "xspot_enable_hyperthreading" {
  description = "Enable hyperthreading in Xspot?"
  type        = bool
  default     = true
}

variable "xspot_enable_balloon" {
  description = "Enable ballooning in Xspot?"
  type        = bool
  default     = true
}

variable "profile_az" {
  description = "Exostellar profile's availability zone."
  type        = string
  default     = ""
}

variable "k8s_node_image_name" {
  description = "Kubernetes node image name."
  type        = string
  # Example: "k8s-132".
  # 
  # Note: Specifying a default value here might cause a mismatch between this and the actual EKS version. Hence, this is
  # made mandatory.
}

variable "xspot_controller_instance_profile_arn" {
  description = "Xspot controller's instance-profile's ARN"
  type        = string
  default     = ""
}

variable "xspot_worker_instance_profile_arn" {
  description = "Xspot worker's instance-profile's ARN"
  type        = string
  default     = ""
}

variable "aws_resource_prefix" {
  description = "Prefix to use for all AWS resources. Note: This will not add any trailing extra characters like '-'."
  type        = string
  default     = ""
}

variable "ssm_enabled" {
  description = "Enable AWS Systems Manager (SSM) for Exostellar Management Server (EMS)?"
  type        = bool
  default     = false
}

variable "xspot_security_group_id" {
  description = "x-spot (controllers and workers) security group ID"
  type        = string
  # TODO :: Bhargav :: Remove the default value after adding the xspot security group change to standalone flow.
  default = ""
}

variable "permissions_boundary_arn" {
  description = "Optional ARN of an IAM policy to use as a permissions boundary for the EMS IAM role created by this module."
  type        = string
  default     = null
}
