locals {
  # Common tags for all resources in this module.
  module_tags = {
    cluster = var.eks_cluster
    region  = var.aws_region
  }
}

# Read the subnet based on its ID.
data "aws_subnet" "cluster_subnet" {
  id = var.subnet_id
}

# Read the IAM policy document for EC2 instance assume role.
data "aws_iam_policy_document" "instance_assume_role_policy" {
  statement {
    actions = ["sts:AssumeRole"]
    principals {
      type        = "Service"
      identifiers = ["ec2.amazonaws.com"]
    }
  }
}

# Read the AWS-managed IAM policy for SSM.
data "aws_iam_policy" "managed_amazon_ssm_managed_instance_core_policy" {
  arn = "arn:aws:iam::aws:policy/AmazonSSMManagedInstanceCore"
}

# Create a security Group for Exostellar Management Server (EMS).
resource "aws_security_group" "exostellar_management_server" {
  name        = "${var.aws_resource_prefix}${var.eks_cluster}-exostellar-management-server-sg"
  description = "Security Group for the Exostellar Management Server"
  vpc_id      = var.vpc_id
  tags = merge(
    local.module_tags,
    {
      Name = "${var.aws_resource_prefix}${var.eks_cluster}-exostellar-management-server-sg",
    }
  )

  # Enable ingress on port 22 (SSH) only if SSM is disabled.
  dynamic "ingress" {
    for_each = var.ssm_enabled ? [] : [1]
    content {
      from_port   = 22
      to_port     = 22
      protocol    = "tcp"
      cidr_blocks = ["0.0.0.0/0"]
    }
  }

  ingress {
    from_port   = 443
    to_port     = 443
    protocol    = "tcp"
    cidr_blocks = ["0.0.0.0/0"]
  }

  ingress {
    from_port   = 5000
    to_port     = 5000
    protocol    = "tcp"
    cidr_blocks = [var.vpc_cidr_block]
  }

  ingress {
    from_port = 0
    to_port   = 0
    protocol  = "-1"
    self      = true
  }

  egress {
    from_port   = 0
    to_port     = 0
    protocol    = "-1"
    cidr_blocks = ["0.0.0.0/0"]
  }
}

# Create IAM Role for Exostellar Management Server (EMS).
resource "aws_iam_role" "exostellar_management_server" {
  name                 = "${var.aws_resource_prefix}${var.eks_cluster}-exostellar-management-server-role"
  assume_role_policy   = data.aws_iam_policy_document.instance_assume_role_policy.json
  permissions_boundary = var.permissions_boundary_arn
  tags                 = local.module_tags
}

# Create IAM Policy for Exostellar Management Server (EMS).
resource "aws_iam_role_policy" "exostellar_management_server" {
  name   = "${var.aws_resource_prefix}${var.eks_cluster}-exostellar-management-server-policy"
  role   = aws_iam_role.exostellar_management_server.id
  policy = file("${path.module}/policy/exostellar-management-server.json")
}

# Attach the SSM policy to the EMS role, only if SSM flag is enabled.
resource "aws_iam_role_policy_attachment" "exostellar_management_server_role_managed_amazon_ssm_managed_instance_core" {
  count = var.ssm_enabled ? 1 : 0

  role       = aws_iam_role.exostellar_management_server.name
  policy_arn = data.aws_iam_policy.managed_amazon_ssm_managed_instance_core_policy.arn
}

# Create IAM Instance Profile for Exostellar Management Server (EMS).
resource "aws_iam_instance_profile" "exostellar_management_server" {
  name = "${var.aws_resource_prefix}${var.eks_cluster}-exostellar-management-server-profile"
  role = aws_iam_role.exostellar_management_server.name
  tags = local.module_tags
}

# Create the Exostellar Management Server (EMS) EC2 instance.
resource "aws_instance" "exostellar_management_server" {
  ami                         = var.region_ami_map[var.aws_region]
  instance_type               = var.instance_type
  subnet_id                   = var.subnet_id
  iam_instance_profile        = aws_iam_instance_profile.exostellar_management_server.name
  associate_public_ip_address = var.is_subnet_public
  vpc_security_group_ids = concat(
    [aws_security_group.exostellar_management_server.id],
    var.shared_security_group_ids != [] ? var.shared_security_group_ids : [],
    var.nfs_security_group_id != "" ? [var.nfs_security_group_id] : [],
  )

  # If SSM is enabled, don't use the SSH key pair.
  key_name = var.ssm_enabled ? "" : var.ssh_key_name

  root_block_device {
    volume_size           = var.volume_size
    delete_on_termination = var.volume_delete_on_termination
    encrypted             = var.encrypt_volume
    volume_type           = var.volume_type
  }

  tags = merge(
    local.module_tags,
    {
      Name = "${var.aws_resource_prefix}${var.eks_cluster}-exostellar-management-server",
    },
  )
  volume_tags = merge(
    local.module_tags,
    {
      Name = "${var.aws_resource_prefix}${var.eks_cluster}-exostellar-management-server",
    },
  )


  # Flag disable_api_termination, if set to true (i.e., when termination protection is enabled), will prevent accidental
  # termination of the Exostellar Management Server (EMS) EC2 instance over the AWS console or API.
  disable_api_termination = var.termination_protection

  # Enable force deletion of the Exostellar Management Server (EMS) EC2 instance, even if `disable_api_termination` is
  # enabled.
  #
  # Starting from terraform-provider-aws v6.8.0 (see PR below for more info), apart from API and console, the EC2
  # termination protection is enforced on Terraform as well. This means `terraform destroy` will fail unless termination
  # protection on the EMS instance is manually disabled by the user.
  #
  # Reference: https://github.com/hashicorp/terraform-provider-aws/pull/43722
  #
  # However, in terraform-exostellar-modules, we only expect termination protection to guard against accidental
  # deletions through the AWS console or API, not from Terraform-managed workflows. To preserve the expected behavior
  # where `terraform destroy` succeeds even when termination protection is enabled, enable force deletion.
  #
  # Since this is an expected behavior, and was working that way earlier (before PR #43722), force_destroy is hardcoded
  # without exposing it to the user to set.
  force_destroy = true

  # Pass the user data, base64 encoded, to the EMS, to configure EMS options and create import files for EMS.
  user_data_base64 = base64encode(
    <<-EOF
    #!/bin/bash
    cat <<EOT >> /xcompute/ems-options
      DOMAIN_NAME=${var.domain_name}
      NFS_REMOTE_HOST=${var.nfs_dns_name}
    EOT

    mkdir -p /xcompute/import/
    cat > /xcompute/import/profile_overrides.json <<JSON
    ${templatefile("${path.module}/import/profile_overrides.json", {
    profile_az                            = var.profile_az,
    profile_name                          = "az1",
    xspot_controller_instance_profile_arn = var.xspot_controller_instance_profile_arn,
    xspot_controller_security_group_ids = concat(
      [
        aws_security_group.exostellar_management_server.id,
      ],
      # The x-spot security group can be empty depending upon the allow_xspot_worker_inbound_traffic flag in the upper
      # layers. Add it to the list of security groups only if it is not empty.
      #
      # Note: The x-spot security group is only required for workers. But since Karpenter shares the
      # securityGroupSelectorTerms with both controllers and workers, we consider the security group as common for all
      # x-spot components. For consistency, adding it to the controller here as well.
      var.xspot_security_group_id != "" ? [var.xspot_security_group_id] : [],
      var.shared_security_group_ids,
    ),
    xspot_controller_subnet_id        = var.xspot_controller_subnet_id,
    aws_region                        = var.aws_region,
    xspot_worker_instance_profile_arn = var.xspot_worker_instance_profile_arn,
    xspot_worker_subnet_id            = var.xspot_worker_subnet_id,
    xspot_worker_security_group_ids = concat(
      [
        aws_security_group.exostellar_management_server.id,
      ],
      # The x-spot security group can be empty depending upon the allow_xspot_worker_inbound_traffic flag in the upper
      # layers. Add it to the list of security groups only if it is not empty.
      var.xspot_security_group_id != "" ? [var.xspot_security_group_id] : [],
      var.shared_security_group_ids,
    ),
    xspot_enable_hyperthreading = var.xspot_enable_hyperthreading,
    xspot_enable_balloon        = var.xspot_enable_balloon,
    volume_size                 = var.volume_size
    })}
    JSON

    cat > /xcompute/import/env_overrides.json <<JSON
    ${templatefile("${path.module}/import/env_overrides.json", {
    k8s_node_image_name = var.k8s_node_image_name
})}
    JSON
    EOF
)
}
