# Exostellar's Existing Cluster Flow Module

Sets up the Exostellar environment on an existing EKS cluster.

It performs the following actions:

1. Reads the EKS cluster's state.
1. Reads the EKS cluster's auth info.
1. Reads the VPC details.
1. Reads the EKS cluster's TLS certificate details.
1. Reads the subnets (both public + private) in the VPC.
1. Reads each subnet's details.
1. Reads the route table's details for subnets (both public + private).
1. Updates kubeconfig to use the specified EKS cluster.
1. Performs prechecks.
1. Deploys the Exostellar's IAM resources.
1. Deploys the xspot security group.
1. Deploys the Exostellar Management Server (EMS) and related resources.
1. Configures the EKS cluster according to Exostellar’s standard setup.
1. Deploys IRSA for the EBS CSI driver.
1. Adds license to Exostellar Management Server (EMS).
1. Deploys Exostellar's Karpenter (xKarpenter) and related resources.

# Prerequisites

1. The following binaries must be available in the local environment:
    1. `aws` CLI. Refer to [this page](https://docs.aws.amazon.com/cli/latest/userguide/getting-started-install.html)
    for details on installing it.
2. AWS CLI login. Ref. to the [doc](https://docs.aws.amazon.com/signin/latest/userguide/command-line-sign-in.html).
3. A functional EKS cluster and configured in kube config (`~/.kube/config`).
4. Optional SSH key-pair whose name can be specified in var `ssh_key_name` in [main.tf](main.tf).

# Usage

1. Import the module:

    ```hcl
    module "existing_cluster_flow" {
      source  = "git::ssh://git@github.com/Exostellar/terraform-exostellar-modules//modules/existing-cluster-full?ref=v0.0.5"

      eks_cluster  = "my-exostellar-cluster"
      aws_region   = "us-east-1"
      ems_ami_id   = "ami-XXXXXXXXXXXXXXXXX"
      ssh_key_name = "my-ssh-key-pair-name"
    }
    ```
    > Note: This is a minimal example (with just mandatory fields). For full list of inputs and other useful details,
    > refer to example [main.tf](main.tf).

2. Prerequisite Configurations:

    - Configure AWS CNI:

        1. Prevent AWS CNI from running on `x-compute` nodes by adding the following condition to DaemonSet. Refer to
           doc for more details and steps: [Configure-AWS-CNI.md](../../docs/Configure-AWS-CNI.md)

            ```
            {"key": "eks.amazonaws.com/nodegroup","operator": "NotIn","values": ["x-compute"]}
            ```

        2. Disable `AWS_VPC_K8S_CNI_CUSTOM_NETWORK_CFG` on container `aws-node` in AWS CNI DaemonSet. Refer to doc for
           more details and steps: [Configure-AWS-CNI.md](../../docs/Configure-AWS-CNI.md)

    - Configure AWS CSI:

        1. Prevent AWS CSI from running on `x-compute` nodes by adding the following condition to DaemonSet. Refer to
           doc for more details and steps: [Configure-AWS-CSI.md](../../docs/Configure-AWS-CSI.md)

            ```
            {"key": "eks.amazonaws.com/nodegroup","operator": "NotIn","values": ["x-compute"]}
            ```

        2. Annotate EBS CSI driver's service account with IAM role ARN for IRSA. And restart the EBS CSI driver add-on
           or pods (if installed using Helm chart). Refer to doc for more details and steps:
           [Configure-AWS-CSI.md](../../docs/Configure-AWS-CSI.md)

            ```
            eks.amazonaws.com/role-arn: arn:aws:iam::<aws-account-id>:role/<cluster-name>-ebs-csi
            ```

3. To deploy the existing-cluster-flow, run the following commands:

    ```bash
    terraform init
    terraform plan -input=false
    terraform apply -auto-approve
    ```

4. To destroy the existing-cluster-flow, run the following command:

    ```bash
    terraform destroy -auto-approve -refresh=true
    ```

6. (Optional) Revert the prerequisite configurations from step-2.

## Requirements

| Name | Version |
|------|---------|
| [terraform](https://github.com/hashicorp/terraform) | ~> 1.8 |

## Providers

| Name | Version |
|------|---------|
| [aws](https://registry.terraform.io/providers/hashicorp/aws/6.12.0) | 6.12.0 |
| [helm](https://registry.terraform.io/providers/hashicorp/helm/3.0.2) | 3.0.2 |
| [http](https://registry.terraform.io/providers/hashicorp/http/3.5.0) | 3.5.0 |
| [kubernetes](https://registry.terraform.io/providers/hashicorp/kubernetes/2.38.0) | 2.38.0 |
| [null](https://registry.terraform.io/providers/hashicorp/null/3.2.4) | 3.2.4 |
| [tls](https://registry.terraform.io/providers/hashicorp/tls/4.1.0) | 4.1.0 |

## Modules

| Name | Source | Version |
|------|--------|---------|
| [iam](../../modules/iam) | ../../modules/iam | n/a |
| [ems](../../modules/ems/) | ../../modules/ems | n/a |
| [karpenter](../../modules/karpenter/) | ../../modules/karpenter | n/a |

## Resources

No resources (or) data sources are needed apart from the `existing_cluster_flow` module.

## Inputs

### 1. EKS Cluster Inputs

| Name | Description | Type | Default | Required |
|------|-------------|------|---------|:--------:|
| aws_region | AWS region for deploying resources. Eg. 'us-east-1'. | `string` | `null` | yes |
| eks_cluster | EKS cluster's name. It must be between 1 and 26 characters in length. | string | `null` | yes |

### 2. Exostellar Management Server (EMS) Inputs

| Name | Description | Type | Default | Required |
|------|-------------|------|---------|:--------:|
| domain_name | Domain name to run Exostellar Management Server (EMS) in. Optional. | string | `""` | no |
| ems_ami_id | Mapping of AWS region to Exostellar Management Server (EMS)'s AMI. | string | `null` | yes |
| ems_instance_type | EC2 instance type for Exostellar Management Server (EMS). | string | `"m5d.xlarge"` | no |
| ems_termination_protection | Enable termination protection for Exostellar Management Server (EMS)'s EC2 instance? | bool | `true` | no |
| ems_volume_delete_on_termination | Delete the Exostellar Management Server (EMS) instance's root volume on termination of instance? | `bool` | `true` | no |
| ems_volume_encryption | Encrypt the Exostellar Management Server (EMS) instance's root volume? | `bool` | `false` | no |
| ems_volume_size_gb | Exostellar Management Server (EMS) instance's root volume size (in GB). | number | `100` | no |
| ems_volume_type | Exostellar Management Server (EMS) instance's root volume type. Valid options: standard, gp2, gp3, io1, io2, sc1, and st1. | `string` | `"gp2"` | no |
| license_filepath | The Exostellar's license file name/path. | string | `""` | no |
| nfs_dns_name | DNS name of the NFS file system for Exostellar Management Server (EMS). | string | `""` | no |
| nfs_security_group_id | Security group ID that enable traffic between the NFS server and Exostellar Management Server (EMS). | string | `""` | no |
| ssh_key_name | SSH key-pair's name in AWS for Exostellar Management Server (EMS)'s access. If ssm_enabled is set to true, this will be skipped. | string | `null` | no |

### 3. Xspot Inputs

| Name | Description | Type | Default | Required |
|------|-------------|------|---------|:--------:|
| allow_xspot_worker_inbound_traffic | Enable inbound traffic to x-spot worker? | `bool` | `true` | no |
| enable_infrastructure_optimizer | Enable Infrastructure Optimizer (IO) for xspot? | `bool` | `true` | no |
| enable_workload_optimizer | Enable Workload Optimizer (WO) for xspot? | `bool` | `true` | no |
| xspot_enable_balloon | Enable ballooning in Xspot? | bool | `true` | no |
| xspot_enable_hyperthreading | Enable hyperthreading in Xspot? | bool | `true` | no |
| xspot_version | Xspot (controller and worker) version to use. | `string` | `"xspot-3.4.0"` | no |

### 4. CNI and CSI Inputs

| Name | Description | Type | Default | Required |
|------|-------------|------|---------|:--------:|
| exo_cni_chart_namespace | Exostellar's CNI chart's namespace. | `string` | `"kube-system"` | no |
| exo_cni_chart_repository | Exostellar's CNI chart's repository URL. | `string` | `"oci://public.ecr.aws/u8h5n6o4/exostellar-cni/exo-aws-vpc-cni"` | no |
| exo_cni_chart_version | Exostellar's CNI chart's version. | `string` | `"1.20.0"` | no |
| exo_csi_chart_namespace | Exostellar's CSI chart's namespace. | `string` | `"kube-system"` | no |
| exo_csi_chart_repository | Exostellar's CSI chart's repository URL. | `string` | `"oci://public.ecr.aws/u8h5n6o4/exostellar-csi/exo-aws-ebs-csi-driver"` | no |
| exo_csi_chart_version | Exostellar's CSI chart's version. | `string` | `"1.46.0"` | no |

### 5. Exostellar Karpenter (xKarpenter) Inputs

| Name | Description | Type | Default | Required |
|------|-------------|------|---------|:--------:|
| guest_kernel_version | Guest kernel version for xspot nodes. | `string` | `"5.15.185"` |
| pod_resource_limits | Pod resource requests and limits. | <pre>map(<br>  object(<br>    {<br>      cpu    = number<br>      memory = string<br>    },<br>  ),<br>) | <pre>{<br>  requests = {<br>    cpu    = 1<br>    memory = "1Gi"<br>  }<br>  limits = {<br>    cpu    = 1<br>    memory = "1Gi"<br>  }<br>} | no |
| xkarpenter_helm_chart_repository | Exostellar's Karpenter (xkarpenter)'s Helm chart's repository URL. | string | `"oci://public.ecr.aws/u8h5n6o4/exostellar-karpenter/karpenter"` | no |
| xkarpenter_namespace | Exostellar's Karpenter (xkarpenter)'s namespace. | string | `"exokarpenter"` | no |
| xkarpenter_resources_helm_chart_repository | Exostellar's Karpenter (xkarpenter)'s resources (ExoNodeClass and ExoNodePool) Helm chart's repository URL. | string | `"oci://public.ecr.aws/u8h5n6o4/exostellar-karpenter/karpenter-resources"` | no |
| xkarpenter_version | Exostellar's Karpenter (xkarpenter)'s Helm chart's version. Note: To keep xKarpenter and xKarpenter Resources compatible, same version is used with both charts. | string | `"v2.0.6"` | no |

### 6. Miscellaneous Inputs

| Name | Description | Type | Default | Required |
|------|-------------|------|---------|:--------:|
| aws_resource_prefix | Prefix to use for all AWS resources. Note: This will not add any trailing extra characters like '-'. | `string` | `""` | no |
| ssm_enabled | Enable AWS Systems Manager (SSM) for Exostellar Management Server (EMS)? | `false` | `bool` | no |

## Outputs

| Name | Description |
|------|-------------|
| exostellar_management_server_console_admin_password | Exostellar Management Server (EMS) console's initial admin password for login. Change on first login. |
| exostellar_management_server_console_admin_username | Exostellar Management Server (EMS) console's admin username for login. |
| exostellar_management_server_private_ip | Exostellar Management Server (EMS)'s private IP. |
| exostellar_management_server_public_ip | Exostellar Management Server (EMS)'s public IP. |
